% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmer_pi_unstruc.R
\name{lmer_pi_unstruc}
\alias{lmer_pi_unstruc}
\title{Prediction intervals for future observations based on linear random effects models}
\usage{
lmer_pi_unstruc(
  model,
  newdat = NULL,
  m = NULL,
  alternative = "both",
  alpha = 0.05,
  nboot = 10000,
  delta_min = 0.01,
  delta_max = 10,
  tolerance = 0.001,
  traceplot = TRUE,
  n_bisec = 30,
  algorithm = "MS22"
)
}
\arguments{
\item{model}{a random effects model of class lmerMod}

\item{newdat}{a \code{data.frame} with the same column names as the historical data
on which the model depends}

\item{m}{number of future observations}

\item{alternative}{either "both", "upper" or "lower". \code{alternative} specifies
if a prediction interval or an upper or a lower prediction limit should be computed}

\item{alpha}{defines the level of confidence (1-\code{alpha})}

\item{nboot}{number of bootstraps}

\item{delta_min}{lower start value for bisection}

\item{delta_max}{upper start value for bisection}

\item{tolerance}{tolerance for the coverage probability in the bisection}

\item{traceplot}{if \code{TRUE}: Plot for visualization of the bisection process}

\item{n_bisec}{maximal number of bisection steps}

\item{algorithm}{either "MS22" or "MS22mod" (see details)}
}
\value{
\code{lmer_pi_futvec()} returns an object of class \code{c("predint", "normalPI")}
with prediction intervals or limits in the first entry (\code{$prediction}).
}
\description{
\code{lmer_pi_unstruc()} calculates a bootstrap calibrated prediction interval for one or more
future observation(s) based on linear random effects models as described in section
3.2.4. of Menssen and Schaarschmidt (2022).
Please note, that the bootstrap calibration used here does not consider the sampling
structure of the future data, since the calibration values are drawn randomly from
bootstrap data sets that have the same structure as the historical data.
}
\details{
This function returns bootstrap-calibrated prediction intervals as well as
lower or upper prediction limits.

If \code{algorithm} is set to "MS22", both limits of the prediction interval
are calibrated simultaneously using the algorithm described in Menssen and
Schaarschmidt (2022), section 3.2.4. The calibrated prediction interval is given
as

\deqn{[l,u] = \hat{\mu} \pm q^{calib} \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1}
\hat{\sigma}^2_c}}

with \eqn{\hat{\mu}} as the expected future observation (historical mean) and
\eqn{\hat{\sigma}^2_c} as the \eqn{c=1, 2, ..., C} variance components and \eqn{\hat{\sigma}^2_{C+1}}
as the residual variance obtained from the random
effects model fitted with \code{lme4::lmer()} and \eqn{q^{calib}} as the as the bootstrap-calibrated
coefficient used for interval calculation. \cr

If \code{algorithm} is set to "MS22mod", both limits of the prediction interval
are calibrated independently from each other. The resulting prediction interval
is given by

\deqn{[l,u] = \Big[\hat{\mu} - q^{calib}_l \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1} \hat{\sigma}^2_c}, \text{   }
\hat{\mu} + q^{calib}_u \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1} \hat{\sigma}^2_c} \Big].}

Please note, that this modification does not affect the calibration procedure, if only
prediction limits are of interest. \cr

This function is an direct implementation of the PI given in Menssen and Schaarschmidt
2022 section 3.2.4.
}
\examples{
# loading lme4
library(lme4)

# Fitting a random effects model based on c2_dat1
fit <- lmer(y_ijk~(1|a)+(1|b)+(1|a:b), c2_dat1)
summary(fit)

# Prediction interval using c2_dat2 as future data
\donttest{pred_int <- lmer_pi_unstruc(model=fit, newdat=c2_dat2, alternative="both", nboot=100)
summary(pred_int)}

# Upper prediction limit for m=3 future observations
\donttest{pred_u <- lmer_pi_unstruc(model=fit, m=3, alternative="upper", nboot=100)
summary(pred_u)}

# Please note that nboot was set to 100 in order to decrease computing time
# of the example. For a valid analysis set nboot=10000.

}
\references{
Menssen and Schaarschmidt (2022): Prediction intervals for all of M future
observations based on linear random effects models. Statistica Neerlandica,
 \doi{10.1111/stan.12260}
}
