% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pareto_frontier.R
\name{pareto_frontier}
\alias{pareto_frontier}
\title{Calculate travel time and monetary cost Pareto frontier}
\usage{
pareto_frontier(
  r5r_core,
  origins,
  destinations,
  mode = c("WALK", "TRANSIT"),
  mode_egress = "WALK",
  departure_datetime = Sys.time(),
  time_window = 1L,
  percentiles = 50L,
  max_walk_time = Inf,
  max_bike_time = Inf,
  max_car_time = Inf,
  max_trip_duration = 120L,
  fare_structure = NULL,
  fare_cutoffs = -1L,
  walk_speed = 3.6,
  bike_speed = 12,
  max_rides = 3,
  max_lts = 2,
  n_threads = Inf,
  verbose = FALSE,
  progress = FALSE,
  output_dir = NULL
)
}
\arguments{
\item{r5r_core}{An object to connect with the R5 routing engine, created with
\code{\link[=setup_r5]{setup_r5()}}.}

\item{origins, destinations}{Either a \verb{POINT sf} object with WGS84 CRS, or a
\code{data.frame} containing the columns \code{id}, \code{lon} and \code{lat}.}

\item{mode}{A character vector. The transport modes allowed for access,
transfer and vehicle legs of the trips. Defaults to \code{WALK}. Please see
details for other options.}

\item{mode_egress}{A character vector. The transport mode used after egress
from the last public transport. It can be either \code{WALK}, \code{BICYCLE} or
\code{CAR}. Defaults to \code{WALK}. Ignored when public transport is not used.}

\item{departure_datetime}{A POSIXct object. Please note that the departure
time only influences public transport legs. When working with public
transport networks, please check the \code{calendar.txt} within your GTFS feeds
for valid dates. Please see details for further information on how
datetimes are parsed.}

\item{time_window}{An integer. The time window in minutes for which \code{r5r}
will calculate multiple travel time matrices departing each minute.
Defaults to 1 minute. By default, the function returns the result based on
median travel times, but the user can set the \code{percentiles} parameter to
extract more results. Please read the time window vignette for more details
on its usage \code{vignette("time_window", package = "r5r")}}

\item{percentiles}{An integer vector (max length of 5). Specifies the
percentile to use when returning travel time estimates within the given
time window. Please note that this parameter is applied to the travel time
estimates only (e.g. if the 25th percentile is specified, and the output
between A and B is 15 minutes and 10 dollars, 25\% of all trips cheaper
than 10 dollars taken between these points are shorter than 15 minutes).
Defaults to 50, returning the median travel time. If a vector with length
bigger than 1 is passed, the output contains an additional column that
specifies the percentile of each travel time and monetary cost
combination. Due to upstream restrictions, only 5 percentiles can be
specified at a time. For more details, please see R5 documentation at
\url{https://docs.conveyal.com/analysis/methodology#accounting-for-variability}.}

\item{max_walk_time}{An integer. The maximum walking time (in minutes) to
access and egress the transit network, or to make transfers within the
network. Defaults to no restrictions, as long as \code{max_trip_duration} is
respected. The max time is considered separately for each leg (e.g. if
you set \code{max_walk_time} to 15, you could potentially walk up to 15 minutes
to reach transit, and up to \emph{another} 15 minutes to reach the destination
after leaving transit). Defaults to \code{Inf}, no limit.}

\item{max_bike_time}{An integer. The maximum cycling time (in minutes) to
access and egress the transit network. Defaults to no restrictions, as long
as \code{max_trip_duration} is respected. The max time is considered separately
for each leg (e.g. if you set \code{max_bike_time} to 15 minutes, you could
potentially cycle up to 15 minutes to reach transit, and up to \emph{another} 15
minutes to reach the destination after leaving transit). Defaults to \code{Inf},
no limit.}

\item{max_car_time}{An integer. The maximum driving time (in minutes) to
access and egress the transit network. Defaults to no restrictions, as long
as \code{max_trip_duration} is respected. The max time is considered separately
for each leg (e.g. if you set \code{max_car_time} to 15 minutes, you could
potentially drive up to 15 minutes to reach transit, and up to \emph{another} 15
minutes to reach the destination after leaving transit). Defaults to \code{Inf},
no limit.}

\item{max_trip_duration}{An integer. The maximum trip duration in minutes.
Defaults to 120 minutes (2 hours).}

\item{fare_structure}{A fare structure object, following the convention
set in \code{\link[=setup_fare_structure]{setup_fare_structure()}}. This object describes how transit fares
should be calculated. Please see the fare structure vignette to
understand how this object is structured:
\code{vignette("fare_structure", package = "r5r")}.}

\item{fare_cutoffs}{A numeric vector. The monetary cutoffs that
should be considered when calculating the Pareto frontier. Most of the
time you'll want this parameter to be the combination of all possible
fares listed in you \code{fare_structure}. Choosing a coarse distribution of
cutoffs may result in many different trips falling within the same cutoff.
For example, if you have two different routes in your GTFS, one costing $3
and the other costing $4, and you set this parameter to \code{5}, the output
will tell you the fastest trips that costed up to $5, but you won't be
able to identify which route was used to complete such trips. In this
case, it would be more beneficial to set the parameter as \code{c(3, 4)} (you
could also specify combinations of such values, such as 6, 7, 8 and so on,
because a transit user could hypothetically benefit from making transfers
between the available routes).}

\item{walk_speed}{A numeric. Average walk speed in km/h. Defaults to 3.6
km/h.}

\item{bike_speed}{A numeric. Average cycling speed in km/h. Defaults to 12
km/h.}

\item{max_rides}{An integer. The maximum number of public transport rides
allowed in the same trip. Defaults to 3.}

\item{max_lts}{An integer between 1 and 4. The maximum level of traffic
stress that cyclists will tolerate. A value of 1 means cyclists will only
travel through the quietest streets, while a value of 4 indicates cyclists
can travel through any road. Defaults to 2. Please see details for more
information.}

\item{n_threads}{An integer. The number of threads to use when running the
router in parallel. Defaults to use all available threads (Inf).}

\item{verbose}{A logical. Whether to show \code{R5} informative messages when
running the function. Defaults to \code{FALSE} (please note that in such case
\code{R5} error messages are still shown). Setting \code{verbose} to \code{TRUE} shows
detailed output, which can be useful for debugging issues not caught by
\code{r5r}.}

\item{progress}{A logical. Whether to show a progress counter when running
the router. Defaults to \code{FALSE}. Only works when \code{verbose} is set to
\code{FALSE}, so the progress counter does not interfere with \code{R5}'s output
messages. Setting \code{progress} to \code{TRUE} may impose a small penalty for
computation efficiency, because the progress counter must be synchronized
among all active threads.}

\item{output_dir}{Either \code{NULL} or a path to an existing directory. When not
\code{NULL} (the default), the function will write one \code{.csv} file with the
results for each origin in the specified directory. In such case, the
function returns the path specified in this parameter. This parameter is
particularly useful when running estimates on memory-constrained settings,
because writing the results to disk prevents \code{r5r} from storing them in
memory.}
}
\value{
A \code{data.table} with the travel time and monetary cost Pareto frontier
between the specified origins and destinations. An additional column
identifying the travel time percentile is present if more than one value
was passed to \code{percentiles}. Origin and destination pairs whose trips
couldn't be completed within the maximum travel time using less money than
the specified monetary cutoffs are not returned in the \code{data.table}. If
\code{output_dir} is not \code{NULL}, the function returns the path specified in
that parameter, in which the \code{.csv} files containing the results are
saved.
}
\description{
Fast computation of travel time and monetary cost Pareto frontier between
origin and destination pairs.
}
\section{Transport modes}{


\code{R5} allows for multiple combinations of transport modes. The options
include:
\itemize{
\item \strong{Transit modes:} \code{TRAM}, \code{SUBWAY}, \code{RAIL}, \code{BUS}, \code{FERRY}, \code{CABLE_CAR},
\code{GONDOLA}, \code{FUNICULAR}. The option \code{TRANSIT} automatically considers all
public transport modes available.
\item \strong{Non transit modes:} \code{WALK}, \code{BICYCLE}, \code{CAR}, \code{BICYCLE_RENT},
\code{CAR_PARK}.
}
}

\section{Level of Traffic Stress (LTS)}{


When cycling is enabled in \code{R5} (by passing the value \code{BIKE} to either
\code{mode} or \code{mode_egress}), setting \code{max_lts} will allow cycling only on
streets with a given level of danger/stress. Setting \code{max_lts} to 1, for
example, will allow cycling only on separated bicycle infrastructure or
low-traffic streets and routing will revert to walking when traversing any
links with LTS exceeding 1. Setting \code{max_lts} to 3 will allow cycling on
links with LTS 1, 2 or 3. Routing also reverts to walking if the street
segment is tagged as non-bikable in OSM (e.g. a staircase), independently of
the specified max LTS.

The default methodology for assigning LTS values to network edges is based
on commonly tagged attributes of OSM ways. See more info about LTS in the
original documentation of R5 from Conveyal at
\url{https://docs.conveyal.com/learn-more/traffic-stress}. In summary:
\itemize{
\item \strong{LTS 1}: Tolerable for children. This includes low-speed, low-volume
streets, as well as those with separated bicycle facilities (such as
parking-protected lanes or cycle tracks).
\item \strong{LTS 2}: Tolerable for the mainstream adult population. This includes
streets where cyclists have dedicated lanes and only have to interact with
traffic at formal crossing.
\item \strong{LTS 3}: Tolerable for "enthused and confident" cyclists. This includes
streets which may involve close proximity to moderate- or high-speed
vehicular traffic.
\item \strong{LTS 4}: Tolerable only for "strong and fearless" cyclists. This
includes streets where cyclists are required to mix with moderate- to
high-speed vehicular traffic.
}

For advanced users, you can provide custom LTS values by adding a tag \verb{<key = "lts">} to the \code{osm.pbf} file.
}

\section{Datetime parsing}{


\code{r5r} ignores the timezone attribute of datetime objects when parsing dates
and times, using the study area's timezone instead. For example, let's say
you are running some calculations using Rio de Janeiro, Brazil, as your study
area. The datetime \code{as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S")} will be parsed as May 13th, 2019, 14:00h in
Rio's local time, as expected. But \code{as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S", tz = "Europe/Paris")} will also be parsed as
the exact same date and time in Rio's local time, perhaps surprisingly,
ignoring the timezone attribute.
}

\section{Routing algorithm}{


The \code{\link[=detailed_itineraries]{detailed_itineraries()}} and \code{\link[=pareto_frontier]{pareto_frontier()}} functions use an
\code{R5}-specific extension to the McRAPTOR routing algorithm. The
implementation used in \code{detailed_itineraries()} allows the router to find
paths that are optimal and less than optimal in terms of travel time, with
some heuristics around multiple access modes, riding the same patterns, etc.
The specific extension to McRAPTOR to do suboptimal path routing is not
documented yet, but a detailed description of base McRAPTOR can be found in
Delling et al (2015). The implementation used in \code{pareto_frontier()}, on the
other hand, returns only the fastest trip within a given monetary cutoff,
ignoring slower trips that cost the same. A detailed discussion on the
algorithm can be found in Conway and Stewart (2019).
\itemize{
\item Delling, D., Pajor, T., & Werneck, R. F. (2015). Round-based public
transit routing. Transportation Science, 49(3), 591-604.
\doi{10.1287/trsc.2014.0534}
\item Conway, M. W., & Stewart, A. F. (2019). Getting Charlie off the MTA: a
multiobjective optimization method to account for cost constraints in public
transit accessibility metrics. International Journal of Geographical
Information Science, 33(9), 1759-1787. \doi{10.1080/13658816.2019.1605075}
}
}

\examples{
\dontshow{if (identical(tolower(Sys.getenv("NOT_CRAN")), "true")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(r5r)

# build transport network
data_path <- system.file("extdata/poa", package = "r5r")
r5r_core <- setup_r5(data_path = data_path)

# load origin/destination points
points <- read.csv(file.path(data_path, "poa_hexgrid.csv"))[1:5,]

# load fare structure object
fare_structure_path <- system.file(
  "extdata/poa/fares/fares_poa.zip",
  package = "r5r"
)
fare_structure <- read_fare_structure(fare_structure_path)

departure_datetime <- as.POSIXct(
  "13-05-2019 14:00:00",
  format = "\%d-\%m-\%Y \%H:\%M:\%S"
)

pf <- pareto_frontier(
  r5r_core,
  origins = points,
  destinations = points,
  mode = c("WALK", "TRANSIT"),
  departure_datetime = departure_datetime,
  fare_structure = fare_structure,
  fare_cutoffs = c(4.5, 4.8, 9, 9.3, 9.6)
)
head(pf)

stop_r5(r5r_core)
\dontshow{\}) # examplesIf}
}
\seealso{
Other routing: 
\code{\link{detailed_itineraries}()},
\code{\link{expanded_travel_time_matrix}()},
\code{\link{travel_time_matrix}()}
}
\concept{routing}
