load_target <- function(src, local = TRUE){
  s <- source(src, local = local, echo = FALSE, verbose = FALSE, chdir = TRUE)
  target <- s$value
  if(!is.list(target)){
    stop("Script ", src, " must need to end with a list.")
  }
  target
}



#' @rdname rave-pipeline
#' @export
load_targets <- function(..., env = NULL){
  if(!is.environment(env)) {
    env <- TRUE
  }
  targets <- lapply(c(...), load_target, local = env)
  do.call("c", targets)
}

# Activate pipeline within an environment
# Must be called within a function
# Must not be called directory
activate_pipeline <- function(pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
                              debug = FALSE) {
  if(!dir.exists(pipe_dir)){
    stop("`pipe_dir` must be the directory of pipeline files.")
  }

  pipe_dir <- normalizePath(pipe_dir)
  common_script <- file.path(pipe_dir, "common.R")
  entry_point_path <- file.path(pipe_dir, "make-main.R")

  if(!file.exists(common_script)){
    stop("Cannot find `common.R` in the pipeline.")
  }

  if(!file.exists(entry_point_path)){
    stop("Cannot find entry point (`make-main.R`) in the pipeline.")
  }

  parent_frame <- parent.frame()
  current <- Sys.getenv("TAR_PROJECT", "main")

  # R doesn't like it when the working directory is normalized
  wd0 <- getwd()
  wd <- normalizePath(wd0)

  if( debug && !interactive() ) {
    stop("Debugging a pipeline must be in an interactive session.")
  }

  setwd(pipe_dir)
  # Handle reset work directory to comply with CRAN policy


  if( debug ) {
    # Shouldn't set working directory back since this is explicitly requested
    # However, warning should be raised and method to set back is provided
    # Also `interactive()` has just been checked.
    warning(sprintf("Debugging a pipeline. Current working directory has been altered. Please run the following script once debug is finished.\n  setwd('%s');Sys.setenv('TAR_PROJECT' = '%s')", wd, current))
  } else {

    do.call(on.exit, list(
      bquote({
        Sys.setenv("TAR_PROJECT" = .(current))
        setwd(.(wd0))
      }), add = TRUE, after = TRUE
    ), envir = parent_frame)

  }

  tmpenv <- new.env(parent = globalenv())
  source("common.R", local = tmpenv)

  # Sys.setenv("RAVE_PIPELINE" = pipe_dir)

  attr(pipe_dir, "target_name") <- tmpenv$target_name
  attr(pipe_dir, "target_script") <- tmpenv$target_script
  attr(pipe_dir, "target_directory") <- tmpenv$target_directory
  pipe_dir
}

#' @rdname rave-pipeline
#' @export
pipeline_target_names <- function(pipe_dir = Sys.getenv("RAVE_PIPELINE", ".")){
  pipe_dir <- activate_pipeline(pipe_dir)

  # find targets that are not in the main
  script <- attr(pipe_dir, "target_script")

  all_targets <- load_target("make-main.R")
  target_names <- unlist(lapply(all_targets, function(x){
    x$settings$name
  }))
  target_names
}

#' @rdname rave-pipeline
#' @export
pipeline_debug <- function(
    quick = TRUE,
    env = parent.frame(),
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    skip_names
){
  pipe_dir <- activate_pipeline(pipe_dir)

  # find targets that are not in the main
  script <- attr(pipe_dir, "target_script")

  main_targets <- load_target(script)
  all_targets <- load_target("make-main.R")

  if(quick){
    if(missing(skip_names)){
      main_target_names <- unlist(lapply(main_targets, function(x){
        x$settings$name
      }))
      target_names <- unlist(lapply(all_targets, function(x){
        x$settings$name
      }))
      skip_names <- unname(target_names[!target_names %in% main_target_names])
    }
  } else {
    skip_names <- NULL
  }

  if(quick){
    # build with targets
    do.call(targets::tar_make, list(
      callr_function = NULL,
      envir = env, names = as.call(
        list(quote(targets::all_of), skip_names)
      )
    ))
  }

  nms <- names(all_targets)

  w <- getOption("width", 80)

  started <- Sys.time()

  for(ii in seq_along(all_targets)){
    if(length(nms) < ii || nms[[ii]] == ""){
      nm <- "(No name)"
    } else {
      nm <- strsplit(nms[[ii]], "_", fixed = TRUE)[[1]]
      nm[[1]] <- str_to_sentence(nm[[1]])
      nm <- paste(nm, collapse = " ")
    }

    ...t <- all_targets[[ii]]
    name <- ...t$settings$name
    if(name %in% skip_names){
      v <- targets::tar_read_raw(name)
      assign(name, v, envir = env)
    } else {

      r <- w - nchar(nm, type = "width") - 14
      if( r <= 2 ){ r <- 2 }
      nm <- paste(c(
        sprintf(" (%.2f s) ", time_delta(started, Sys.time())),
        rep("-", r), " ", nm, "\n"), collapse = "")
      catgl(nm, level = "INFO")

      counter <- Sys.time()

      {
        message("Evaluating -> ", name, "\r", appendLF = FALSE)
        v <- eval(...t$command$expr, new.env(parent = env))
        assign(name, v, envir = env)
        str <- deparse1(v)
        str <- gsub(x = str, pattern = "[\t\n]", replacement = "  ")
        r <- w - nchar(name, type = "width") - 25
        if(r < 0){
          r <- max(w - 5, 1)
          s <- "`{name}` <- \n    "
        } else {
          s <- "`{name}` <- "
        }
        str <- substr(str, start = 1, stop = r)
        delta <- time_delta(counter, Sys.time())
        catgl(sprintf("[+%6.2f s] ", delta), s, str, "\n")
        counter <- Sys.time()
      }
    }
  }
}

#' @rdname rave-pipeline
#' @export
pipeline_dep_targets <- function(
    names, skip_names = NULL,
    pipe_dir = Sys.getenv("RAVE_PIPELINE", ".")) {

  force(names)

  pipe_dir <- activate_pipeline(pipe_dir)
  # script <- attr(pipe_dir, "target_script")
  # main_targets <- load_target(script)
  all_targets <- load_target("make-main.R")

  dep_env <- new.env(parent = emptyenv(), hash = TRUE)

  tnames <- vapply(all_targets, function(target) {
    target$settings$name
  }, "", USE.NAMES = FALSE)

  tparents <- structure(
    lapply(all_targets, function(target) {
      target$command$deps
    }),
    names = tnames
  )

  walk <- function(name) {
    if(!is.null(dep_env[[name]])) {
      dep_env[[name]] <- dep_env[[name]] + 1L
      return()
    }
    dep_env[[name]] <- 1L
    pa <- tparents[[name]]
    lapply(pa, walk)
    return()
  }

  lapply(names, walk)
  re <- ls(dep_env, all.names = TRUE)

  if(length(skip_names)) {
    dep_env <- new.env(parent = emptyenv(), hash = TRUE)
    lapply(skip_names, walk)
    skipped_names <- ls(dep_env, all.names = TRUE)
    re <- re[!re %in% skipped_names]
    attr(re, "skipped") <- skipped_names
  }
  re
}

#' @rdname rave-pipeline
#' @export
pipeline_eval <- function(names, env = new.env(parent = parent.frame()),
                          pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
                          settings_path = file.path(pipe_dir, "settings.yaml"),
                          shortcut = FALSE) {

  force(env)
  pipe_dir <- activate_pipeline(pipe_dir)


  # DIPSAUS DEBUG START
  # self <- ravepipeline::pipeline('import_lfp_native')
  # env = new.env(parent = parent.frame())
  # pipe_dir = self$pipeline_path
  # settings_path = file.path(pipe_dir, "settings.yaml")
  # pipe_dir <- activate_pipeline(pipe_dir, debug = TRUE)
  # names <- NULL

  # find targets that are not in the main
  script <- attr(pipe_dir, "target_script")

  main_targets <- load_target(script)
  all_targets <- load_target("make-main.R")

  nms <- names(all_targets)
  tnames <- unname(unlist(lapply(all_targets, function(t){ t$settings$name })))
  if(is.null(names)) {
    names <- tnames
  }
  names <- names[names %in% tnames]

  # Load shared functions into env
  env$pipeline <- pipeline_from_path(pipe_dir)
  skip_load_shared_env <- isTRUE(get0(".is_rave_pipeline_shared_env", envir = env, ifnotfound = FALSE))
  if(!skip_load_shared_env) {
    shared_libs <- list.files(file.path(pipe_dir, "R"), pattern = "^shared-.*\\.R",
                              full.names = TRUE, ignore.case = TRUE)
    shared_libs <- sort(shared_libs)

    lapply(shared_libs, function(f) {
      catgl(sprintf("Loading script - %s", gsub("^.*[/|\\\\]shared-", "shared-", f)), .envir = emptyenv(), level = "DEFAULT")
      source(file = f, local = env, chdir = TRUE)
      return()
    })
  }

  # if(dir.exists(file.path(pipe_dir, "py"))) {
  #   pipeline_py_module(pipe_dir = pipe_dir,
  #                      convert = FALSE)
  # }

  input_names <- NULL
  if(file.exists(settings_path)) {
    catgl(sprintf("Loading inputs from [%s]", basename(settings_path)), .envir = emptyenv(), level = "DEFAULT")
    input_settings <- read_yaml(settings_path)
    input_settings <- input_settings[names(input_settings) %in% tnames]
    if(length(input_settings)) {
      input_names <- names(input_settings)
      lapply(input_names, function(nm) {
        env[[nm]] <- resolve_pipeline_settings_value( input_settings[[nm]], pipe_dir = pipe_dir )
      })
    }
  }
  # print(ls(env))

  if( shortcut ) {
    matured_targets <- ls(env, all.names = TRUE, sorted = FALSE)
    if(!isTRUE(env$.is_rave_pipeline_shared_env)) {
      penv <- parent.env(env)
      if(is.environment(penv) && isTRUE(penv$.is_rave_pipeline_shared_env)) {
        matured_targets <- c(matured_targets, ls(penv, all.names = TRUE, sorted = FALSE))
      }
    }
  } else {
    matured_targets <- NULL
  }
  missing_names <- tnames[!tnames %in% c(names, matured_targets, input_names)]

  w <- getOption("width", 80)
  all_starts <- Sys.time()

  missing_key <- new.env()

  meta <- targets::tar_meta(store = targets::tar_config_get("store"))

  eval_target <- function(name) {

    # determine the name
    ii <- which(tnames == name)
    tar_obj <- all_targets[[ii]]
    if(length(nms) < ii || nms[[ii]] == ""){
      nm <- sprintf("[%s]", name)
    } else {
      nm <- strsplit(nms[[ii]], "_")[[1]]
      nm[[1]] <- str_to_sentence(nm[[1]])
      nm <- paste(nm, collapse = " ")
      nm <- sprintf("[%s] (%s)", name, nm)
    }
    readable_name <- nm

    if( shortcut && name %in% missing_names ) {
      v <- pipeline_read(var_names = name, ifnotfound = missing_key, meta = meta)
      if(!identical(v, missing_key)) {
        env[[name]] <- v
        matured_targets <<- c(matured_targets, name)
        catgl(sprintf("Loaded - %s", readable_name), .envir = emptyenv(), level = "DEFAULT")
        return()
      }
    }

    deps <- tar_obj$command$deps
    deps <- deps[!deps %in% matured_targets]
    if( length(deps) ) {
      lapply(deps, eval_target)
    }

    if( name %in% matured_targets ) { return() }

    started <- Sys.time()
    catgl(sprintf("Starting - %s ...", readable_name), .envir = emptyenv(), level = "DEFAULT")
    v <- eval(tar_obj$command$expr, new.env(parent = env))
    assign(name, v, envir = env)
    matured_targets <<- c(matured_targets, name)

    ended <- Sys.time()

    msg <- sprintf(
      "Evaluated - %s [%.2f sec, %s] - %s",
      name, time_delta(started, ended, units = "secs"),
      to_ram_size(utils::object.size(v)),
      paste(class(v), collapse = ", ")
    )
    catgl(msg, .envir = emptyenv(), level = "DEFAULT")

    return()

  }

  lapply(names, eval_target)
  #
  #   if(length(missing_names)) {
  #     list2env(
  #       pipeline_read(var_names = missing_names),
  #       envir = env
  #     )
  #   }
  #
  #   w <- getOption("width", 80)
  #
  #   all_starts <- Sys.time()
  #
  #   lapply(names, function(name) {
  #
  #     ii <- which(tnames == name)
  #     if(length(nms) < ii || nms[[ii]] == ""){
  #       nm <- sprintf("[%s]", name)
  #     } else {
  #       nm <- strsplit(nms[[ii]], "_")[[1]]
  #       nm[[1]] <- str_to_sentence(nm[[1]])
  #       nm <- paste(nm, collapse = " ")
  #       nm <- sprintf("[%s] (%s)", name, nm)
  #     }
  #
  #     nm <- paste(c(
  #       sprintf(" (%.2f s) ", time_delta(all_starts, Sys.time())),
  #       rep("-", 2), " ", nm, "\n"), collapse = "")
  #     catgl(nm, level = "INFO")
  #
  #     tar_obj <- all_targets[[ii]]
  #     started <- Sys.time()
  #     v <- eval(tar_obj$command$expr, new.env(parent = env))
  #     assign(name, v, envir = env)
  #     ended <- Sys.time()
  #
  #     msg <- sprintf(
  #       "%s [%.2f sec, %s] - %s",
  #       name, time_delta(started, ended, units = "secs"),
  #       to_ram_size(utils::object.size(v)),
  #       paste(class(v), collapse = ", ")
  #     )
  #     catgl(msg, .envir = emptyenv(), level = "DEFAULT")
  #     NULL
  #   })
  env
}

# May be removed later if not really useful
pipeline_run_interactive <- function(
    names, skip_names, env = parent.frame(),
    pipe_dir = Sys.getenv("RAVE_PIPELINE", ".")
){
  pipe_dir <- activate_pipeline(pipe_dir)

  # find targets that are not in the main
  script <- attr(pipe_dir, "target_script")

  main_targets <- load_target(script)
  all_targets <- load_target("make-main.R")

  main_target_names <- unlist(lapply(main_targets, function(x){
    x$settings$name
  }))
  target_names <- unlist(lapply(all_targets, function(x){
    x$settings$name
  }))
  if(missing(skip_names)){
    skip_names <- unname(target_names[!target_names %in% main_target_names])
  }

  if(length(skip_names)){
    # build with targets
    do.call(targets::tar_make, list(
      callr_function = NULL,
      envir = env, names = skip_names
    ))
    list2env(
      pipeline_read(var_names = skip_names, pipe_dir = pipe_dir, simplify = FALSE),
      envir = env
    )
  }

  w <- getOption("width", 80)
  started <- Sys.time()

  nms <- names(all_targets)

  for(nm in names){
    ii <- which(target_names == nm)
    if(length(ii)){

      desc <- nms[[ii]]
      if(desc == "") {
        desc <- "(No name)"
      } else {
        desc <- strsplit(nms[[ii]], "_", perl = TRUE)[[1]]
        desc[[1]] <- str_to_sentence(desc[[1]])
        desc <- paste(desc, collapse = " ")
      }

      ...t <- all_targets[[ii]]
      name <- ...t$settings$name
      r <- w - nchar(nm, type = "width") - 14
      if( r <= 2 ){ r <- 2 }
      nm <- paste(c(
        sprintf(" (%.2f s) ", time_delta(started, Sys.time())),
        rep("-", r), " ", nm, "\n"), collapse = "")
      catgl(nm, level = "INFO")

      counter <- Sys.time()
      {
        message("Evaluating -> ", name, "\r", appendLF = FALSE)
        expr <- ...t$command$expr
        tryCatch({
          v <- eval(expr, new.env(parent = env))
          assign(name, v, envir = env)
          str <- deparse1(v)
          str <- gsub(x = str, pattern = "\t|\n", replacement = "  ")
          r <- w - nchar(name, type = "width") - 25
          if(r < 0){
            r <- max(w - 5, 1)
            s <- "`{name}` <- \n    "
          } else {
            s <- "`{name}` <- "
          }
          str <- substr(str, start = 1, stop = r)
          delta <- time_delta(counter, Sys.time())
          catgl(sprintf("[+%6.2f s] ", delta), s, str, "\n")
        }, error = function(e){
          e$call <- expr
          stop(e)
        })
        counter <- Sys.time()
      }
    }

  }
}

#' @rdname rave-pipeline
#' @export
pipeline_visualize <- function(
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    glimpse = FALSE, targets_only = TRUE,
    shortcut = FALSE, zoom_speed = 0.1, ...
){
  pipe_dir <- activate_pipeline(pipe_dir)
  if(glimpse) {
    targets::tar_glimpse(targets_only = targets_only, shortcut = shortcut,
                         zoom_speed = zoom_speed, ...)
  } else {
    targets::tar_visnetwork(targets_only = targets_only, shortcut = shortcut,
                            zoom_speed = zoom_speed, ...)
  }
}


pipeline_dependency_graph <- function(pipeline_path, targets_only = TRUE, shortcut = FALSE,
                                      zoom_speed = 0.1, aspect_ratio = 1.5, main = "",
                                      node_size = 30, label_size = node_size, glimpse = FALSE) {

  require_package("visNetwork")

  widget <- callr::r(
    function(pipeline_path, targets_only, shortcut, zoom_speed, aspect_ratio, main, label_size, node_size, glimpse) {
      ravepipeline <- asNamespace("ravepipeline")
      targets <- asNamespace("targets")
      visNetwork <- asNamespace("visNetwork")

      pipeline_path <- ravepipeline$activate_pipeline(pipeline_path)

      target_names <- ravepipeline$pipeline_target_names(pipeline_path)
      target_descr <- sapply(strsplit(names(target_names), "_"), function(x){
        x <- x[x != ""]
        if(!length(x)) { return(NA) }
        substr(x[[1]], start = 1, stop = 1) <- toupper(
          substr(x[[1]], start = 1, stop = 1)
        )
        paste(x, collapse = " ")
      })
      descr <- data.frame(
        name = target_names,
        rave_description = target_descr
      )

      target_script <- attr(pipeline_path, "target_script")
      # load & combine pipelines
      target <- ravepipeline$load_targets(target_script)
      target <- targets$pipeline_init(target)

      store <- targets$tar_config_get("store")
      names <- targets$pipeline_get_names(target)
      reporter <- targets$tar_config_get("reporter_outdated")


      if( glimpse ) {
        network <- targets$glimpse_init(
          pipeline = target, meta = targets$meta_init(path_store = store),
          progress = targets$progress_init(path_store = store), targets_only = targets_only,
          names = names, shortcut = shortcut, allow = NULL,
          exclude = ".Random.seed")
      } else {
        network <- targets$inspection_init(
          pipeline = target, meta = targets$meta_init(path_store = store),
          progress = targets$progress_init(path_store = store), targets_only = targets_only,
          names = names, shortcut = shortcut, allow = NULL,
          exclude = ".Random.seed", outdated = TRUE, reporter = reporter)
      }
      visual <- targets$visnetwork_init(
        network = network, label = NULL,
        level_separation = 100, degree_from = 1L,
        degree_to = 1L, zoom_speed = zoom_speed)


      level <- visual$network$vertices$level
      height <- max(c(table(level), 1))
      width <- max(c(level, 0)) + 1

      visual$level_separation <- height / width * 150 * aspect_ratio
      visual$update_network()
      visual$update_labels()
      visual$update_colors()
      visual$update_extra()
      visual$update_legend()

      vertices <- merge(visual$network$vertices, descr, by = "name", all.x = TRUE, all.y = FALSE)
      vertices$shape <- "hexagon"
      vertices$title <- sprintf(
        "Variable: %s%s%s",
        vertices$label,
        ifelse(is.na(vertices$rave_description), "", sprintf("<br>Description: %s", vertices$rave_description)),
        ifelse(is.na(vertices$bytes), "", sprintf("<br>Size:     %.1f MB", vertices$bytes / 1024))
      )
      edges <- visual$network$edges
      out <- visNetwork$visNetwork(nodes = vertices, edges = edges, main = main)
      out <- visNetwork$visNodes(out, physics = FALSE, size = node_size, font = list(size = label_size))
      out <- visNetwork$visEdges(out, smooth = list(type = "cubicBezier",
                                                    forceDirection = "horizontal"))
      out <- visNetwork$visOptions(
        graph = out, collapse = TRUE,
        highlightNearest = list(
          enabled = TRUE, algorithm = "hierarchical",
          degree = list(from = min(visual$degree_from, nrow(vertices)),
                        to = min(visual$degree_to, nrow(vertices)))))
      out <- visNetwork$visLegend(graph = out, useGroups = FALSE, enabled = !glimpse,
                                  addNodes = visual$legend, ncol = 1L, position = "right", width = 0.1,
                                  zoom = FALSE)
      out <- visNetwork$visPhysics(graph = out, stabilization = FALSE)
      out <- visNetwork$visInteraction(graph = out, zoomSpeed = visual$zoom_speed)
      widget <- visNetwork$visHierarchicalLayout(
        edgeMinimization = TRUE,
        graph = out, direction = "LR", levelSeparation = visual$level_separation, sortMethod = "directed"
      )
      visual$visual <- widget
      return( widget )
    },
    args = list(
      pipeline_path = pipeline_path, targets_only = targets_only,
      shortcut = shortcut, zoom_speed = zoom_speed, aspect_ratio = aspect_ratio,
      main = main, node_size = node_size, label_size = label_size, glimpse = glimpse
    )
  )

  return(widget)
}


#' @rdname rave-pipeline
#' @export
pipeline_progress <- function(
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    method = c("summary", "details", "custom"),
    func = targets::tar_progress_summary
){
  method <- match.arg(method)
  activate_pipeline(pipe_dir)

  switch (
    method,
    "summary" = targets::tar_progress_summary(),
    "details" = targets::tar_progress(),
    {
      func()
    }
  )
}

#' @rdname rave-pipeline
#' @export
pipeline_fork <- function(
    src = Sys.getenv("RAVE_PIPELINE", "."),
    dest = tempfile(pattern = "rave_pipeline_"),
    policy = "default",
    activate = FALSE,
    ...
){

  if(!dir.exists(src)){
    stop("pipeline_fork: `src` must be a pipeline directory")
  }
  if(!file.exists(file.path(src, "common.R"))){
    stop("pipeline_fork: `src/common.R` is missing")
  }
  if(!file.exists(file.path(src, "make-main.R"))){
    stop("pipeline_fork: `src/make-main.R` is missing")
  }

  # search for fork-policy
  fork_policy_path <- file.path(src, "fork-policy")
  fork_policy_regexp <- c("^shared", "^main\\.html$")
  if(file.exists(fork_policy_path)) {
    fork_policy <- readLines(fork_policy_path)
    idx <- which(startsWith(fork_policy, "["))
    if(length(idx)) {
      sel <- which(tolower(trimws(fork_policy[idx])) == sprintf("[%s]", tolower(policy)))
      if(length(sel)) {
        sel <- sel[[1]]
        if(sel == length(idx)) {
          start <- idx[[sel]] + 1
          end <- length(fork_policy)
          if( start <= end ) {
            fork_policy_regexp <- fork_policy[seq(start, end)]
          }
        } else {
          start <- idx[[sel]] + 1
          end <- idx[[sel + 1]] - 1
          if( start <= end ) {
            fork_policy_regexp <- fork_policy[seq(start, end)]
          }
        }
      }
    }
  }
  # clean up
  fork_policy_regexp <- fork_policy_regexp[nzchar(trimws(fork_policy_regexp))]

  # ignore .files
  fork_policy_regexp <- c(fork_policy_regexp, "(^\\.|/\\.)")

  # list all the files
  fs <- list.files(
    src,
    include.dirs = FALSE,
    full.names = FALSE,
    ignore.case = TRUE,
    all.files = TRUE,
    recursive = TRUE,
    no.. = TRUE
  )
  # format backslashes
  fs <- gsub("[/|\\\\]+", "/", fs)

  ignore_files <- lapply(fork_policy_regexp, function(regexp) {
    fs[grepl(regexp, x = fs, ignore.case = TRUE, perl = TRUE)]
  })
  ignore_files <- unique(unlist(ignore_files))
  fs <- fs[!fs %in% ignore_files]

  dir_create2(dest)
  dest <- normalizePath(dest, mustWork = TRUE)

  # Copy the files
  lapply(fs, function(file) {
    src_path <- file.path(src, file)
    dst_path <- file.path(dest, file)
    dir_create2(dirname(dst_path))
    file.copy(
      from = src_path,
      to = dst_path,
      overwrite = TRUE,
      recursive = FALSE,
      copy.date = TRUE
    )
  })

  # gather fork information
  fork_info <- tryCatch({
    descr <- get_module_description(dest)
    list(
      policy = policy,
      version = descr$Version
    )
  }, error = function(e) {
    list(
      policy = policy,
      version = "0.0.0.9000"
    )
  })
  saveRDS(fork_info, file.path(dest, "_fork_info"))

  if( activate ){
    pipeline_build(dest)
    Sys.setenv("RAVE_PIPELINE" = dest)
  }
  dest
}

#' @rdname rave-pipeline
#' @export
pipeline_build <- function(
    pipe_dir = Sys.getenv("RAVE_PIPELINE", ".")
){
  pipe_dir <- activate_pipeline(pipe_dir)
  configure_src <- file.path(pipe_dir, "configure.R")

  # build _targets.yaml
  source(configure_src, local = TRUE, chdir = TRUE)

  TRUE
}

#' @rdname rave-pipeline
#' @export
pipeline_read <- function(
    var_names,
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    branches = NULL,
    ifnotfound = NULL,
    dependencies = c('none', 'ancestors_only', 'all'),
    simplify = TRUE,
    ...
) {
  dependencies <- match.arg(dependencies)

  if( dependencies != "none") {
    dep_names <- pipeline_dep_targets(names = var_names, pipe_dir = pipe_dir)
    if( dependencies == "ancestors_only" ) {
      dep_names <- dep_names[!dep_names %in% var_names]
    }
    var_names <- dep_names
    simplify <- FALSE
  }

  pipe_dir <- activate_pipeline(pipe_dir)

  args <- list(...)
  meta <- args$meta
  if(!is.data.frame(meta)) {
    meta <- targets::tar_meta(store = targets::tar_config_get("store"))
  }

  re <- structure(lapply(var_names, function(vn){
    tryCatch({
      targets::tar_read_raw(name = vn, branches = branches, meta = meta)
    }, error = function(e){
      ifnotfound
    })
  }), names = var_names)

  if(simplify && length(var_names) == 1){
    re <- re[[ 1 ]]
  }
  return(re)
}

#' @rdname rave-pipeline
#' @export
pipeline_vartable <- function(
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    targets_only = TRUE,
    complete_only = FALSE,
    ...
) {
  pipe_dir <- activate_pipeline(pipe_dir)
  tbl <- NULL
  tryCatch({
    if( targets::tar_exist_meta() ) {
      tbl <- targets::tar_meta(..., targets_only = targets_only,
                               complete_only = complete_only)
    }
  }, error = function(e){
  })
  tbl
}

#' @rdname rave-pipeline
#' @export
pipeline_hasname <- function(
    var_names,
    pipe_dir = Sys.getenv("RAVE_PIPELINE", ".")
) {
  tbl <- pipeline_vartable(pipe_dir = pipe_dir)
  var_names %in% tbl$name
}

#' @rdname rave-pipeline
#' @export
pipeline_watch <- function(
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    targets_only = TRUE, ...
){
  pipe_dir <- activate_pipeline(pipe_dir)
  targets::tar_watch(..., targets_only = targets_only, display = 'graph')
}

#' @rdname rave-pipeline
#' @export
pipeline_create_template <- function(
    root_path, pipeline_name, overwrite = FALSE,
    activate = TRUE, template_type = c("rmd", 'r', 'rmd-bare', 'rmd-scheduler', 'rmd-python')
) {
  # DIPSAUS DEBUG START
  # root_path = tempfile()
  # pipeline_name = "junk"
  # overwrite = FALSE
  # activate = TRUE
  # template_type = 'rmd-python'
  template_type <- match.arg(template_type)
  pipeline_name <- tolower(pipeline_name)
  stopifnot2(!pipeline_name %in% c("main", "imports", "initialize", "template"),
             msg = "Pipeline name cannot be `main`, `imports`, `template`, or `initialize`")
  pipe_path <- file.path(root_path, pipeline_name)
  if(dir.exists(pipe_path)){
    if(!overwrite){
      stop("Pipeline ", pipeline_name, " already exists at\n  ", pipe_path)
    } else {
      unlink(pipe_path, recursive = TRUE)
    }
  }
  dir_create2(pipe_path)
  pipe_path <- normalizePath(pipe_path)

  # create a skeleton template
  template_foldername <- switch(
    template_type,
    "rmd-python" = "template-py",
    {
      sprintf("template-%s", template_type)
    }
  )
  template_path <- system.file("rave-pipelines", template_foldername, package = 'ravepipeline', mustWork = TRUE)
  fs_src <- list.files(template_path, recursive = FALSE,
                       include.dirs = TRUE, no.. = TRUE,
                       full.names = FALSE)

  fs_src <- fs_src[!grepl("\\.Rproj$", fs_src, perl = TRUE)]

  for(f in fs_src) {
    f_src <- file.path(template_path, f)
    if(dir.exists(f_src)) {
      # file.copy(f_src, pipe_path, overwrite = overwrite,
      #           recursive = TRUE, copy.date = TRUE)

      # this is a directory, carefully copy all the files (also change the names)
      sub_fs <- list.files(f_src, all.files = FALSE, full.names = FALSE, recursive = TRUE, include.dirs = FALSE, no.. = TRUE)
      for(sub_f in sub_fs) {
        sub_f_src <- file.path(f_src, sub_f)
        # substitude folder names
        sub_f_dst <- gsub(pattern = "TEMPLATE", replacement = pipeline_name, x = file.path(f, sub_f), ignore.case = FALSE)
        sub_f_dst <- file.path(pipe_path, sub_f_dst)
        # make sure the parent directory exists
        dir_create2(dirname(sub_f_dst))
        file.copy(from = sub_f_src, to = sub_f_dst,
                  overwrite = overwrite, copy.date = TRUE)
      }
    } else {
      f_dst <- gsub(x = f, pattern = "TEMPLATE", replacement = pipeline_name, ignore.case = FALSE)
      file.copy(f_src, file.path(pipe_path, f_dst),
                overwrite = overwrite, copy.date = TRUE)
    }
  }
  fs_src <- list.files(template_path, recursive = TRUE, include.dirs = FALSE)
  fs_dst <- gsub(x = fs_src, pattern = "TEMPLATE", replacement = pipeline_name, ignore.case = FALSE)
  fs <- file.path(pipe_path, fs_dst)
  for(f in fs){
    s <- readLines(f)
    s <- gsub(x = s, pattern = "TEMPLATE_PATH", replacement = pipe_path, ignore.case = FALSE)
    s <- gsub(x = s, pattern = "TEMPLATE", replacement = pipeline_name, ignore.case = FALSE)
    s <- gsub(x = s, pattern = "PROJECT_NAME", replacement = "demo", ignore.case = FALSE)
    s <- gsub(x = s, pattern = "SUBJECT_CODE", replacement = "DemoSubject", ignore.case = FALSE)
    writeLines(s, f)
  }
  # settings <- read_yaml(file.path(pipe_path, "settings.yaml"))
  # settings$epoch <- "default"
  # settings$electrodes <- deparse_svec(14L)
  # settings$reference <- "default"

  # save_yaml(settings, file.path(pipe_path, "settings.yaml"))

  # build the pipeline
  pipeline_build(pipe_path)

  if(activate){
    Sys.setenv("RAVE_PIPELINE" = pipe_path)
  }
  return(pipe_path)
}

#' @rdname rave-pipeline
#' @export
pipeline_create_subject_pipeline <- function(
    subject, pipeline_name, overwrite = FALSE,
    activate = TRUE, template_type = c("rmd", 'r', 'rmd-python')
){

  # this function requires a non-CRAN package `raveio`
  # check if this package is installed, otherwise error out
  raveio <- require_package(package = "raveio", return_namespace = TRUE)

  template_type <- match.arg(template_type)
  pipeline_name <- tolower(pipeline_name)
  stopifnot2(!pipeline_name %in% c("main", "imports", "initialize", "template"),
             msg = "Pipeline name cannot be `main`, `imports`, `template`, or `initialize`")
  subject <- raveio$as_rave_subject(subject, strict = FALSE)
  pipe_path <- file.path(subject$pipeline_path, pipeline_name)
  if(!overwrite && dir.exists(pipe_path)){
    stop("Pipeline ", pipeline_name, " already exists at\n  ", pipe_path)
  }
  # create a skeleton template
  template_foldername <- switch(
    template_type,
    "rmd-python" = "template-py",
    {
      sprintf("template-%s", template_type)
    }
  )
  template_path <- system.file("rave-pipelines", template_foldername, package = "ravepipeline", mustWork = TRUE)
  fs_src <- list.files(template_path)
  fs_dst <- gsub(x = fs_src, pattern = "TEMPLATE", replacement = pipeline_name, ignore.case = FALSE)
  dir_create2(pipe_path)
  pipe_path <- normalizePath(pipe_path)
  file.copy(file.path(template_path, fs_src), file.path(pipe_path, fs_dst), overwrite = overwrite, copy.date = TRUE)

  fs <- file.path(pipe_path, fs_dst)
  for(f in fs){
    s <- readLines(f)
    s <- gsub(x = s, pattern = "TEMPLATE_PATH", replacement = pipe_path, ignore.case = FALSE)
    s <- gsub(x = s, pattern = "TEMPLATE", replacement = pipeline_name, ignore.case = FALSE)
    s <- gsub(x = s, pattern = "PROJECT_NAME", replacement = subject$project_name, ignore.case = FALSE)
    s <- gsub(x = s, pattern = "SUBJECT_CODE", replacement = subject$subject_code, ignore.case = FALSE)
    writeLines(s, f)
  }
  settings <- read_yaml(file.path(pipe_path, "settings.yaml"))
  if(length(subject$epoch_names)){
    settings$epoch <- subject$epoch_names[[1]]
  } else {
    settings$epoch <- "default"
  }

  if(length(subject$electrodes)){
    settings$electrodes <- deparse_svec(subject$electrodes)
  }

  if(length(subject$reference_names)){
    settings$reference <- subject$reference_names[[1]]
  } else {
    settings$reference <- "default"
  }

  save_yaml(settings, file.path(pipe_path, "settings.yaml"), sorted = TRUE)

  # build the pipeline
  pipeline_build(pipe_path)

  if(activate){
    Sys.setenv("RAVE_PIPELINE" = pipe_path)
  }
  return(pipe_path)
}

#' @rdname rave-pipeline
#' @export
pipeline_description <- function (file) {
  # file <- file.path(pipeline, 'DESCRIPTION')
  dcf <- read.dcf(file = file)
  if (nrow(dcf) < 1L) {
    stop(sprintf("DESCRIPTION file '%s' is corrupt",
                 file), domain = NA)
  }
  desc <- as.list(dcf[1, ])
  if ((length(desc) == 0)) {
    stop(sprintf("DESCRIPTION file '%s' is missing or broken", file), domain = NA)
  }
  attr(desc, "file") <- file

  class(desc) <- "packageDescription"
  desc
}


#' @name pipeline_settings_get_set
#' @title Get or change pipeline input parameter settings
#' @param key,... the character key(s) to get or set
#' @param default the default value is key is missing
#' @param constraint the constraint of the resulting value; if not \code{NULL},
#' then result must be within the \code{constraint} values, otherwise the
#' first element of \code{constraint} will be returned. This is useful to make
#' sure the results stay within given options
#' @param pipeline_path the root directory of the pipeline
#' @param pipeline_settings_path the settings file of the pipeline, must be
#' a 'yaml' file; default is \code{'settings.yaml'} in the current pipeline
#' @returns \code{pipeline_settings_set} returns a list of all the settings.
#' \code{pipeline_settings_get} returns the value of given key.
#'
#' @examples
#'
#'
#'
#' root_path <- tempfile()
#' pipeline_root_folder <- file.path(root_path, "modules")
#'
#' # create pipeline folder
#' pipeline_path <- pipeline_create_template(
#'   root_path = pipeline_root_folder, pipeline_name = "raveio_demo",
#'   overwrite = TRUE, activate = FALSE, template_type = "rmd-bare")
#'
#' # Set initial user inputs
#' yaml::write_yaml(
#'   x = list(
#'     n = 100,
#'     pch = 16,
#'     col = "steelblue"
#'   ),
#'   file = file.path(pipeline_path, "settings.yaml")
#' )
#'
#' # build the pipeline for the first time
#' # this is a one-time setup
#' pipeline_build(pipeline_path)
#'
#' # get pipeline settings
#' pipeline_settings_get(
#'   key = "n",
#'   pipeline_path = pipeline_path
#' )
#'
#' # get variable with default if missing
#' pipeline_settings_get(
#'   key = "missing_variable",
#'   default = "missing",
#'   pipeline_path = pipeline_path
#' )
#'
#' pipeline_settings_set(
#'   missing_variable = "A",
#'   pipeline_path = pipeline_path
#' )
#'
#' pipeline_settings_get(
#'   key = "missing_variable",
#'   default = "missing",
#'   pipeline_path = pipeline_path
#' )
#'
#'
#' unlink(root_path, recursive = TRUE)
#'
#'
#' @export
pipeline_settings_set <- function(
    ...,
    pipeline_path = Sys.getenv("RAVE_PIPELINE", "."),
    pipeline_settings_path = file.path(pipeline_path, "settings.yaml")
){
  if(!file.exists(pipeline_settings_path)){
    stop("Cannot find settings file:\n  ", pipeline_settings_path)
  }
  settings <- load_yaml(pipeline_settings_path)
  args <- list(...)
  if( !length(args) ) { return(settings) }

  argnames <- names(args)
  if(!length(argnames) || "" %in% argnames) {
    stop("`pipeline_set`: all input lists must have names")
  }


  lapply(argnames, function(nm) {

    opts <- resolve_pipeline_settings_opt(settings[[nm]], strict = FALSE)

    if(!is.null(opts)) {
      # external settings
      # save external data
      pipeline_save_extdata(
        data = args[[nm]],
        name = opts$name,
        format = opts$format,
        overwrite = TRUE,
        pipe_dir = pipeline_path
      )
      return()
    }

    # otherwise replace settings directly
    settings[[nm]] <- args[[nm]]

  })
  # list_to_fastmap2(args, map = settings)
  tf <- tempfile()
  on.exit({ unlink(tf) })
  save_yaml(x = settings, file = tf, sorted = TRUE)
  file.copy(from = tf, to = pipeline_settings_path,
            overwrite = TRUE, recursive = FALSE)
  settings

}


resolve_pipeline_settings_opt <- function(value, strict = TRUE) {

  if(isTRUE(is.character(value)) && length(value) == 1 && !is.na(value) &&
     grepl("^\\$\\{EXTDATA\\-SETTINGS\\|(.*)\\}$", value)) {

    # this value should be stored as external data
    value <- gsub(pattern = "(^\\$\\{EXTDATA\\-SETTINGS\\||\\}$)", "", x = value, ignore.case = FALSE)
    value <- strsplit(value, "\\|")[[1]]
    data_name <- value[[1]]

    if(nchar(data_name) && grepl("[a-zA-Z0-9]{1,}[a-zA-Z0-9_\\.-]{0,}", data_name)) {

      data_format <- "rds"
      if(length(value >= 2) && tolower(value[[2]]) %in% c("json", "yaml", "csv", "fst", "rds")) {
        data_format <- tolower(value[[2]])
      }
      return(list(
        name = data_name,
        format = data_format
      ))
    } else {
      if( strict ) {
        stop("Cannot resolve the pipeline external settings: invalid data name: ", data_name)
      }
      return(NULL)
    }
  } else {
    if( strict ) {
      stop("Cannot resolve the pipeline external settings: invalid settings: ", value)
    }
    return(NULL)
  }
}

resolve_pipeline_settings_value <- function(value, pipe_dir = Sys.getenv("RAVE_PIPELINE", ".")) {

  opts <- resolve_pipeline_settings_opt(value, strict = FALSE)
  if(is.null(opts) || !is.list(opts)) {
    return( value )
  }

  opts$error_if_missing <- FALSE
  opts$default_if_missing <- structure(list(), class = "key_missing")
  opts$pipe_dir <- pipe_dir

  value <- do.call(pipeline_load_extdata, opts)

  if(!is.null(value)) {
    cls <- class(value)
    if( !"raveio-pipeline-extdata" %in% cls ) {
      class(value) <- c("raveio-pipeline-extdata", cls)
    }
    attr(value, "raveio-pipeline-extdata-opts") <- opts[c("name", "format")]
  }
  return( value )
}

#' @rdname pipeline_settings_get_set
#' @export
pipeline_settings_get <- function(
    key, default = NULL, constraint = NULL,
    pipeline_path = Sys.getenv("RAVE_PIPELINE", "."),
    pipeline_settings_path = file.path(pipeline_path, "settings.yaml")) {
  if(!file.exists(pipeline_settings_path)){
    stop("Cannot find settings file:\n  ", pipeline_settings_path)
  }

  settings <- load_yaml(pipeline_settings_path)

  if(missing(key)) {
    nms <- names(settings)
  } else {
    nms <- key
  }


  if(missing(key)){
    lapply(names(settings), function(nm) {
      if(nm != "") {
        settings[[nm]] <- resolve_pipeline_settings_value( settings[[nm]], pipe_dir = pipeline_path )
      }
    })
    return( settings )
  }
  if(!settings$`@has`(key)){
    re <- default
  } else {
    re <- resolve_pipeline_settings_value( settings[[key]], pipe_dir = pipeline_path )
    if(inherits(re, "key_missing")) {
      re <- default
    }
  }

  if(length(constraint)){
    re <- re %OF% constraint
  }
  re

}


#' @rdname rave-pipeline
#' @export
pipeline_load_extdata <- function(
    name, format = c("auto", "json", "yaml", "csv", "fst", "rds"),
    error_if_missing = TRUE, default_if_missing = NULL,
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."), ...
) {
  path <- file.path(pipe_dir, "data")
  format <- match.arg(format)
  if(format == "auto") {
    fs <- list.files(path, recursive = FALSE)
    name2 <- sprintf("%s.%s", name, c("json", "yaml", "csv", "fst", "rds"))
    fs <- fs[fs %in% name2]
  } else {
    fs <- sprintf("%s.%s", name, format)
  }
  if(!length(fs)) {
    if( error_if_missing ) {
      stop("Pipeline: data [", name, "] is missing")
    } else {
      return(default_if_missing)
    }
  }
  fs <- fs[[1]]
  file <- file.path(path, fs)

  if(!file.exists(file)) {
    if( error_if_missing ) {
      stop("Pipeline: data [", name, "] is missing")
    } else {
      return(default_if_missing)
    }
  }

  ext <- strsplit(file, "\\.")[[1]]
  ext <- tolower(ext[[length(ext)]])

  re <- tryCatch({
    switch(
      ext,
      "json" = { load_json(con = file, ...) },
      "yaml" = { load_yaml(file = file, ...) },
      "csv" = { utils::read.csv(file = file, ...) },
      "fst" = { load_fst(path = file, ...) },
      "rds" = { readRDS(file = file, ...) },
      { stop("Unsupported file format") }
    )
  }, error = function(e) {
    if( error_if_missing ) {
      stop("Pipeline: cannot load data [", name, "] with given format [", ext, "]. The file format is inconsistent or file is corrupted.")
    } else {
      return(default_if_missing)
    }
  })

  return(re)
}

#' @rdname rave-pipeline
#' @export
pipeline_save_extdata <- function(
    data, name, format = c("json", "yaml", "csv", "fst", "rds"),
    overwrite = FALSE, pipe_dir = Sys.getenv("RAVE_PIPELINE", "."), ...
) {
  format <- match.arg(format)

  # pipe_dir <- activate_pipeline(pipe_dir)
  path <- file.path(pipe_dir, "data")
  path <- dir_create2(path)
  paths <- file.path(path,  sprintf("%s.%s", name, c("json", "yaml", "csv", "fst", "rds")))

  if(any(file.exists(paths))) {
    if( overwrite ) {
      paths <- paths[file.exists(paths)]
      for(f in paths) {
        unlink(f)
      }
    } else {
      stop("Pipeline: Cannot save data because the data name [", name, "] already exists.")
    }
  }
  path <- file.path(path, sprintf("%s.%s", name, format))
  switch(
    format,
    "json" = {
      save_json(x = data, con = path, serialize = TRUE, ...)
    },
    "yaml" = {
      save_yaml(x = data, file = path, sorted = TRUE, ...)
    },
    "csv" = {
      utils::write.csv(x = data, file = path, ...)
    },
    "fst" = {
      save_fst(x = data, path = path, ...)
    },
    "rds" = {
      saveRDS(object = data, file = path, ...)
    },
    { stop("Unsupported file format") }
  )
  invisible(path)
}


#' @rdname rave-pipeline
#' @export
pipeline_shared <- function(pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
                            callr_function = callr::r) {

  pipe_dir <- normalizePath(pipe_dir, mustWork = TRUE)

  impl <- function(pipe_dir) {
    shared_env <- new.env(parent = globalenv())
    runtime_env <- new.env(parent = globalenv())
    ravepipeline <- asNamespace("ravepipeline")
    runtime_env$pipe_dir <- pipe_dir
    runtime_env$shared_env <- shared_env

    with(runtime_env, {
      ravepipeline <- asNamespace("ravepipeline")
      pipe_dir <- ravepipeline$activate_pipeline(pipe_dir)
      target_script <- attr(pipe_dir, "target_script")

      # shared_libs <-
      #   list.files(
      #     file.path(pipe_dir, "R"),
      #     pattern = "^shared-.*\\.R",
      #     full.names = TRUE,
      #     ignore.case = TRUE
      #   )
      #
      #
      # lapply(sort(shared_libs), function(f) {
      #   source(file = f,
      #          local = shared_env,
      #          chdir = TRUE)
      # })

      # load & combine pipelines
      ravepipeline$load_targets(target_script, env = shared_env)

    })

    return(shared_env)
  }

  # try to get environment from targets
  if(is.function(callr_function)) {
    env <- callr_function(
      impl,
      args = list(pipe_dir = pipe_dir),
      cmdargs = c("--slave", "--no-save", "--no-restore")
    )
  } else {
    env <- impl(pipe_dir)
  }

  env$.is_rave_pipeline_shared_env <- TRUE

  return( env )

}


pipeline_py_info <- function(pipe_dir = Sys.getenv("RAVE_PIPELINE", "."), must_work = NA) {

  pipe_dir <- normalizePath(pipe_dir, mustWork = TRUE)
  env_yaml <- file.path(pipe_dir, "py", c("rave-py-submodule.yaml", "rave-py-submodule.yml"))
  env_yaml <- env_yaml[file.exists(env_yaml)]
  if(!length(env_yaml)) {
    msg <- sprintf("Unable to find python sub-module for the pipeline: no `rave-py-submodule.yaml` found. (pipeline: %s)", pipe_dir)
    if(is.na(must_work)) {
      warning(msg)
    } else if (must_work) {
      stop(msg)
    }
    return()
  }

  py_pkg_name <- NULL
  tryCatch({
    env_yaml <- load_yaml(env_yaml[[1]])
    py_pkg_name <- env_yaml$name
  }, error = function(e) {
    stop("Unable to load python sub-module for the pipeline: cannot parse `rave-py-submodule.yaml`")
  })

  if(!length(py_pkg_name)) {
    stop("Unable to find name for python sub-module from `rave-py-submodule.yaml`")
  }

  module_path <- file.path(pipe_dir, "py", py_pkg_name)
  if(!dir.exists(module_path)) {
    stop("Unable to load python sub-module: module [",
         py_pkg_name, "] is not found under the `py` folder!")
  }

  list(
    pipeline_path = pipe_dir,
    module_path = module_path,
    target_path = file.path(module_path, "rave_pipeline_adapters"),
    module_name = py_pkg_name
  )
}

pipeline_py_module <- function(
    pipe_dir = Sys.getenv("RAVE_PIPELINE", "."), must_work = NA,
    convert = FALSE) {

  pipe_dir <- normalizePath(pipe_dir, mustWork = TRUE)
  info <- pipeline_py_info(pipe_dir = pipe_dir, must_work = must_work)

  py_pkg_name <- info$module_name

  cwd <- getwd()
  pydir <- file.path(pipe_dir, "py")
  setwd(pydir)
  on.exit({
    if(length(cwd) == 1) {
      setwd(cwd)
    }
  }, add = TRUE, after = TRUE)

  py_module <- rpymat::import(py_pkg_name, convert = convert, delay_load = FALSE)

  # set it before the function ends
  setwd(cwd)
  cwd <- NULL

  py_module
}


#' @rdname rave-pipeline
#' @export
pipeline_set_preferences <- function(
    ..., .list = NULL,
    .pipe_dir = Sys.getenv("RAVE_PIPELINE", "."),
    .preference_instance = NULL) {
  prefs <- c(list(...), .list)
  if(!length(prefs)) { return(invisible()) }
  # preferences must be `global/module_id`.`type (graphics, ...)`.`key`.dtype
  nms <- names(prefs)
  if(length(nms) != length(prefs) || any(nms == "")) {
    stop("All preferences must be named")
  }

  if(missing(.preference_instance) || is.null(.preference_instance)) {
    pipe_dir <- activate_pipeline(.pipe_dir)
    pipeline_name <- attr(pipe_dir, "target_name")
    instance <- global_preferences(.prefix_whitelist = c("global", pipeline_name))
  } else {
    instance <- .preference_instance
  }

  instance$mset(.list = prefs)

  invisible(prefs)

}

#' @rdname rave-pipeline
#' @export
pipeline_get_preferences <- function(
    keys, simplify = TRUE, ifnotfound = NULL, validator = NULL, ...,
    .preference_instance = NULL) {

  if(missing(.preference_instance) || is.null(.preference_instance)) {
    instance <- global_preferences()
  } else {
    instance <- .preference_instance
  }


  if(is.function(validator)) {
    args <- list(...)
    force(ifnotfound)
    re <- structure(
      names = keys,
      lapply(keys, function(key) {
        if(instance$has(key)) {
          value <- instance$get(key, missing_default = ifnotfound)
          tryCatch({
            do.call(validator, c(list(value), args))
            return(value)
          }, error = function(e) {
            ifnotfound
          })
        } else {
          ifnotfound
        }
      })
    )
  } else {
    re <- instance$mget(keys, missing_default = ifnotfound)
  }
  if(simplify && length(keys) == 1) {
    re <- re[[1]]
  }
  return(re)
}

#' @rdname rave-pipeline
#' @export
pipeline_has_preferences <- function(keys, ..., .preference_instance = NULL) {
  if(missing(.preference_instance) || is.null(.preference_instance)) {
    instance <- global_preferences()
  } else {
    instance <- .preference_instance
  }
  instance$has(keys, ...)
}

