% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RecoSys.R
\name{tune}
\alias{tune}
\title{Tuning Model Parameters}
\arguments{
\item{r}{Object returned by \code{\link{Reco}}().}

\item{train_data}{An object of class "DataSource" that describes the source
of training data, typically returned by function
\code{\link{data_file}()} or \code{\link{data_memory}()}.}

\item{opts}{A number of candidate tuning parameter values and extra options in the
model tuning procedure. See section \strong{Parameters and Options}
for details.}
}
\value{
A list with two components:

\describe{
  \item{\code{min}}{Parameter values with minimum cross validation RMSE. This
                    is a list that can be passed to the \code{opts} argument
                    in \code{$\link{train}()}.}
  \item{\code{res}}{A data frame giving the supplied candidate
                    values of tuning parameters, and one column showing the
                    RMSE associated with each combination.}
}
}
\description{
This method is a member function of class "\code{RecoSys}"
that uses cross validation to tune the model parameters.

The common usage of this method is
\preformatted{r = Reco()
r$tune(train_data, opts = list(dim      = c(10L, 20L),
                               costp_l1 = c(0, 0.1),
                               costp_l2 = c(0.01, 0.1),
                               costq_l1 = c(0, 0.1),
                               costq_l2 = c(0.01, 0.1),
                               lrate    = c(0.01, 0.1))
)}
}
\section{Parameters and Options}{

The \code{opts} argument should be a list that provides the candidate values
of tuning parameters and some other options. For tuning parameters (\code{dim},
\code{costp_l1}, \code{costp_l2}, \code{costq_l1}, \code{costq_l2},
and \code{lrate}), users can provide a numeric vector for each one, so that
the model will be evaluated on each combination of the candidate values.
For other non-tuning options, users should give a single value. If a parameter
or option is not set by the user, the program will use a default one.

See below for the list of available parameters and options:

\describe{
\item{\code{dim}}{Tuning parameter, the number of latent factors.
                  Can be specified as an integer vector, with default value
                  \code{c(10L, 20L)}.}
\item{\code{costp_l1}}{Tuning parameter, the L1 regularization cost for user factors.
                       Can be specified as a numeric vector, with default value
                       \code{c(0, 0.1)}.}
\item{\code{costp_l2}}{Tuning parameter, the L2 regularization cost for user factors.
                       Can be specified as a numeric vector, with default value
                       \code{c(0.01, 0.1)}.}
\item{\code{costq_l1}}{Tuning parameter, the L1 regularization cost for item factors.
                       Can be specified as a numeric vector, with default value
                       \code{c(0, 0.1)}.}
\item{\code{costq_l2}}{Tuning parameter, the L2 regularization cost for item factors.
                       Can be specified as a numeric vector, with default value
                       \code{c(0.01, 0.1)}.}
\item{\code{lrate}}{Tuning parameter, the learning rate, which can be thought
                    of as the step size in gradient descent.
                    Can be specified as a numeric vector, with default value
                    \code{c(0.01, 0.1)}.}
\item{\code{loss}}{Character string, the loss function. Default is "l2", see
                   section \strong{Parameters and Options} in \code{$\link{train}()}
                   for details.}
\item{\code{nfold}}{Integer, the number of folds in cross validation. Default is 5.}
\item{\code{niter}}{Integer, the number of iterations. Default is 20.}
\item{\code{nthread}}{Integer, the number of threads for parallel
                      computing. Default is 1.}
\item{\code{nmf}}{Logical, whether to perform non-negative matrix factorization.
                  Default is \code{FALSE}.}
\item{\code{verbose}}{Logical, whether to show detailed information. Default is
                      \code{FALSE}.}
}
}
\examples{
\dontrun{
train_set = system.file("dat", "smalltrain.txt", package = "recosystem")
train_src = data_file(train_set)
r = Reco()
set.seed(123) # This is a randomized algorithm
res = r$tune(
    train_src,
    opts = list(dim = c(10, 20, 30),
                costp_l1 = 0, costq_l1 = 0,
                lrate = c(0.05, 0.1, 0.2), nthread = 2)
)
r$train(train_src, opts = res$min)
}

}
\author{
Yixuan Qiu <\url{http://statr.me}>
}
\references{
W.-S. Chin, Y. Zhuang, Y.-C. Juan, and C.-J. Lin.
A Fast Parallel Stochastic Gradient Method for Matrix Factorization in Shared Memory Systems.
ACM TIST, 2015.

W.-S. Chin, Y. Zhuang, Y.-C. Juan, and C.-J. Lin.
A Learning-rate Schedule for Stochastic Gradient Methods to Matrix Factorization.
PAKDD, 2015.

W.-S. Chin, B.-W. Yuan, M.-Y. Yang, Y. Zhuang, Y.-C. Juan, and C.-J. Lin.
LIBMF: A Library for Parallel Matrix Factorization in Shared-memory Systems.
Technical report, 2015.
}
\seealso{
\code{$\link{train}()}
}

