% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redist_smc_tidy.R, R/redist-smc.R
\name{redist_smc}
\alias{redist_smc}
\alias{redist.smc}
\title{SMC Redistricting Sampler}
\usage{
redist_smc(
  map,
  nsims,
  counties = NULL,
  compactness = 1,
  constraints = list(),
  resample = TRUE,
  constraint_fn = function(m) rep(0, ncol(m)),
  adapt_k_thresh = 0.975,
  seq_alpha = 0.2 + 0.3 * compactness,
  truncate = (compactness != 1),
  trunc_fn = redist_quantile_trunc,
  pop_temper = 0,
  ref_name = NULL,
  verbose = TRUE,
  silent = FALSE
)

redist.smc(
  adj,
  total_pop,
  nsims,
  ndists,
  counties = NULL,
  pop_tol = 0.01,
  pop_bounds = NULL,
  compactness = 1,
  constraints = list(),
  resample = TRUE,
  constraint_fn = function(m) rep(0, ncol(m)),
  adapt_k_thresh = 0.975,
  seq_alpha = 0.2 + 0.2 * compactness,
  truncate = (compactness != 1),
  trunc_fn = function(x) pmin(x, 0.01 * nsims^0.4),
  pop_temper = 0,
  verbose = TRUE,
  silent = FALSE,
  adjobj,
  popvec,
  popcons
)
}
\arguments{
\item{map}{A \code{\link{redist_map}} object.}

\item{nsims}{The number of samples to draw.}

\item{counties}{A vector containing county (or other administrative or
geographic unit) labels for each unit, which may be integers ranging from 1
to the number of counties, or a factor or character vector.  If provided, the
algorithm will only generate maps which split up to \code{ndists-1} counties.
If no county-split constraint is desired, this parameter should be left blank.}

\item{compactness}{Controls the compactness of the generated districts, with
higher values preferring more compact districts. Must be nonnegative. See the
'Details' section for more information, and computational considerations.}

\item{constraints}{A list containing information on constraints to implement.
See the 'Details' section for more information.}

\item{resample}{Whether to perform a final resampling step so that the
generated plans can be used immediately.  Set this to \code{FALSE} to perform
direct importance sampling estimates, or to adjust the weights manually.}

\item{constraint_fn}{A function which takes in a matrix where each column is
a redistricting plan and outputs a vector of log-weights, which will be
added the the final weights.}

\item{adapt_k_thresh}{The threshold value used in the heuristic to select a
value \code{k_i} for each splitting iteration. Set to 0.9999 or 1 if
the algorithm does not appear to be sampling from the target distribution.
Must be between 0 and 1.}

\item{seq_alpha}{The amount to adjust the weights by at each resampling step;
higher values prefer exploitation, while lower values prefer exploration.
Must be between 0 and 1.}

\item{truncate}{Whether to truncate the importance sampling weights at the
final step by \code{trunc_fn}.  Recommended if \code{compactness} is not 1.
Truncation only applied if \code{resample=TRUE}.}

\item{trunc_fn}{A function which takes in a vector of weights and returns
a truncated vector. If \code{\link[loo]{loo}} package is installed (strongly
recommended), will default to Pareto-smoothed Importance Sampling (PSIS)
rather than naive truncation.}

\item{pop_temper}{The strength of the automatic population tempering.}

\item{ref_name}{a name for the existing plan, which will be added as a
reference plan, or \code{FALSE} to not include the initial plan in the
output. Defaults to the column name of the existing plan.}

\item{verbose}{Whether to print out intermediate information while sampling.
Recommended.}

\item{silent}{Whether to suppress all diagnostic information.}

\item{adj}{An adjacency matrix, list, or object of class
"SpatialPolygonsDataFrame."}

\item{total_pop}{A vector containing the populations of each geographic unit.}

\item{ndists}{The number of districts in each redistricting plan.}

\item{pop_tol}{The desired population constraint.  All sampled districts
will have a deviation from the target district size no more than this value
in percentage terms, i.e., \code{pop_tol=0.01} will ensure districts have
populations within 1\% of the target population.}

\item{pop_bounds}{A numeric vector with three elements \code{c(lower, target, upper)}
providing more precise population bounds for the algorithm. Districts
will have population between \code{lower} and \code{upper}, with a goal of
\code{target}.  If set, overrides \code{popcons}.}

\item{adjobj}{Deprecated, use adj. An adjacency matrix, list, or object of class
"SpatialPolygonsDataFrame."}

\item{popvec}{Deprecated, use total_pop. A vector containing the populations of each geographic unit.}

\item{popcons}{The desired population constraint.  All sampled districts
will have a deviation from the target district size no more than this value
in percentage terms, i.e., \code{popcons=0.01} will ensure districts have
populations within 1\% of the target population.}
}
\value{
\code{redist_smc} returns an object of class
\code{\link{redist_plans}} containing the simulated plans.

\code{redist.smc} returns an object of class \code{redist}, which
is a list containing the following components:
\item{aList}{The adjacency list used to sample}
\item{cdvec}{The matrix of sampled plans. Each row is a geographical unit,
and each column is a sample.}
\item{wgt}{The importance sampling weights, normalized to sum to 1.}
\item{orig_wgt}{The importance sampling weights before resampling or truncation, normalized to have mean 1.}
\item{nsims}{The number of plans sampled.}
\item{pct_dist_parity}{The population constraint.}
\item{compactness}{The compactness constraint.}
\item{counties}{The computed constraint options list (see above).}
\item{maxdev}{The maximum population deviation of each sample.}
\item{total_pop}{The provided vector of unit populations.}
\item{counties}{The provided county vector.}
\item{adapt_k_thresh}{The provided control parameter.}
\item{seq_alpha}{The provided control vector.}
\item{algorithm}{The algorithm used, here \code{"smc"}.}
}
\description{
\code{redist_smc} uses a Sequential Monte Carlo algorithm to
generate nearly independent congressional or legislative redistricting
plans according to contiguity, population, compactness, and administrative
boundary constraints.
}
\details{
This function draws nearly-independent samples from a specific target measure,
controlled by the \code{popcons}, \code{compactness}, \code{constraints}, and
\code{constraint_fn} parameters.

Key to ensuring good performance is monitoring the efficiency of the resampling
process at each SMC stage.  Unless \code{silent=F}, this function will print
out the effective sample size of each resampling step to allow the user to
monitor the efficiency.  If \code{verbose=T} the function will also print
out information on the \eqn{k_i} values automatically chosen and the
acceptance rate (based on the population constraint) at each step.

Higher values of \code{compactness} sample more compact districts;
setting this parameter to 1 is computationally efficient and generates nicely
compact districts.  Values of other than 1 may lead to highly variable
importance sampling weights.  By default these weights are truncated using
\code{\link{redist_quantile_trunc}} to stabilize the resulting estimates, but
if truncation is used, a specific truncation function should probably be
chosen by the user.

The \code{constraints} parameter allows the user to apply several common
redistricting constraints without implementing them by hand. This parameter
is a list, which may contain any of the following named entries:
\itemize{
\item \code{status_quo}: a list with two entries:
\itemize{
\item \code{strength}, a number controlling the tendency of the generated districts
to respect the status quo, with higher values preferring more similar
districts.
\item \code{current}, a vector containing district assignments for
the current map.
}
\item \code{vra}: a list with three entries:
\itemize{
\item \code{strength}, a number controlling the strength of the Voting Rights Act
(VRA) constraint, with higher values prioritizing majority-minority districts
over other considerations.
\item \code{tgts_min}, the target percentage(s) of minority voters in minority
opportunity districts. Defaults to \code{c(0.55)}.
\item \code{min_pop}, A vector containing the minority population of each
geographic unit.
}
\item \code{incumbency}: a list with two entries:
\itemize{
\item \code{strength}, a number controlling the tendency of the generated districts
to avoid pairing up incumbents.
\item \code{incumbents}, a vector of precinct indices, one for each incumbent's
home address.
}
\item \code{vra_old}: a list with five entries, which may be set up using
\code{\link{redist.constraint.helper}}:
\itemize{
\item \code{strength}, a number controlling the strength of the Voting Rights Act
(VRA) constraint, with higher values prioritizing majority-minority districts
over other considerations.
\item \code{tgt_vra_min}, the target percentage of minority voters in minority
opportunity districts. Defaults to 0.55.
\item \code{tgt_vra_other} The target percentage of minority voters in other
districts. Defaults to 0.25, but should be set to reflect the total minority
population in the state.
\item \code{pow_vra}, which controls the allowed deviation from the target
minority percentage; higher values are more tolerant. Defaults to 1.5
\item \code{min_pop}, A vector containing the minority population of each
geographic unit.
}
}

All constraints are fed into a Gibbs measure, with coefficients on each
constraint set by the corresponding \code{strength} parameters.
The \code{status_quo} constraint adds a term measuring the variation of
information distance between the plan and the reference, rescaled to [0, 1].
The \code{vra} constraint takes a list of target minority percentages. It
matches each district to its nearest target percentage, and then applies a
penalty of the form \eqn{\sqrt{max(0, tgt - minpct)}}, summing across
districts. This penalizes districts which are below their target population.
The \code{incumbency} constraint adds a term counting the number of districts
containing paired-up incumbents.
The \code{vra_old} constraint adds a term of the form
\eqn{(|tgtvramin-minpct||tgtvraother-minpct|)^{powvra})}, which
encourages districts to have minority percentages near either \code{tgt_vra_min}
or \code{tgt_vra_other}. This can be visualized with
\code{\link{redist.plot.penalty}}.
}
\examples{
\dontrun{
data(fl25)

fl_map = redist_map(fl25, ndists=3, pop_tol=0.1)

sampled_basic = redist_smc(fl_map, 10000)

sampled_constr = redist_smc(fl_map, 10000, constraints=list(
                                incumbency = list(strength=1000, incumbents=c(3, 6, 25))
                            ))
}

\dontrun{
data(fl25)
data(fl25_adj)
data(fl25_enum)

sampled_basic = redist.smc(fl25_adj, fl25$pop,
                           nsims=10000, ndists=3, pop_tol=0.1)

sampled_constr = redist.smc(fl25_adj, fl25$pop,
                            nsims=10000, ndists=3, pop_tol=0.1,
                            constraints=list(
                                status_quo = list(strength=10, current=fl25_enum$plans[,5118]),
                                incumbency = lsit(strength=1000, incumbents=c(3, 6, 25))
                            ))
}

}
\references{
McCartan, C., & Imai, K. (2020). Sequential Monte Carlo for Sampling Balanced and Compact Redistricting Plans.
Available at \url{https://imai.fas.harvard.edu/research/files/SMCredist.pdf}.

McCartan, C., & Imai, K. (2020). Sequential Monte Carlo for Sampling Balanced and Compact Redistricting Plans.
Available at \url{https://imai.fas.harvard.edu/research/files/SMCredist.pdf}.
}
\concept{simulate}
