% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/environment.R
\name{makeEnvironment}
\alias{makeEnvironment}
\title{Create reinforcement learning environment.}
\usage{
makeEnvironment(class = "custom", discount = 1, ...)
}
\arguments{
\item{class}{[\code{character(1)}] \cr
Class of environment. One of \code{c("custom", "mdp", "gym", "gridworld")}.}

\item{discount}{[\code{numeric(1)} in (0, 1)] \cr Discount factor.}

\item{...}{[\code{any}] \cr Arguments passed on to the specific environment.}
}
\value{
R6 class of class Environment.
}
\description{
This function creates an environment for reinforcement learning.
}
\details{
Use the \code{step} method to interact with the environment.

Note that all states and actions are numerated starting with 0!

For a detailed explanation and more examples
have a look at the vignette "How to create an environment?".
}
\section{Methods}{

\itemize{
\item \code{$step(action)} \cr
Take action in environment.
Returns a list with \code{state}, \code{reward}, \code{done}.
\item \code{$reset()} \cr
Resets the \code{done} flag of the environment and returns an initial state.
Useful when starting a new episode.
\item \code{$visualize()} \cr
Visualizes the environment (if there is a visualization function).
}
}

\section{Environments}{

\itemize{
\item \link{Environment}
\item \link{GymEnvironment}
\item \link{MdpEnvironment}
\item \link{Gridworld}
\item \link{MountainCar}
}
}

\examples{
step = function(self, action) {
  state = list(mean = action + rnorm(1), sd = runif(1))
  reward = rnorm(1, state[[1]], state[[2]])
  done = FALSE
  list(state, reward, done)
}

reset = function(self) {
  state = list(mean = 0, sd = 1)
  state
}

env = makeEnvironment(step = step, reset = reset, discount = 0.9)
env$reset()
env$step(100)

# Create a Markov Decision Process.
P = array(0, c(2, 2, 2))
P[, , 1] = matrix(c(0.5, 0.5, 0, 1), 2, 2, byrow = TRUE)
P[, , 2] = matrix(c(0, 1, 0, 1), 2, 2, byrow = TRUE)
R = matrix(c(5, 10, -1, 2), 2, 2, byrow = TRUE)
env = makeEnvironment("mdp", transitions = P, rewards = R)

env$reset()
env$step(1L)

# Create a Gridworld.
grid = makeEnvironment("gridworld", shape = c(4, 4),
  goal.states = 15, initial.state = 0)
grid$visualize()

\dontrun{
# Create an OpenAI Gym environment.
# Make sure you have Python, gym and reticulate installed.
env = makeEnvironment("gym", gym.name = "MountainCar-v0")

# Take random actions for 200 steps.
env$reset()
for (i in 1:200) {
  action = sample(env$actions, 1)
  env$step(action)
  env$visualize()
}
env$close()
}
}
