#' Compare counts
#'
#' Compare counts generated by counts()
#'
#' @export
#'
#' @param df The input FARS object.
#' @param interval The interval in which to count: months or years.
#' @param what What to count: crashes, fatalities, or people involved.
#' @param where Where to count, a list with up to three elements:
#'    states ("all" by default), region ("all"), urb ("all")
#' @param who The type of person to count: all (default) drivers, passengers,
#'    pedestrians, or bicyclists.
#' @param involved Factors involved with the crash. Can be any of: distracted
#'     driver, police pursuit, motorcycle, pedalcyclist,
#'     bicyclist, pedestrian, pedbike, young driver, older driver, speeding,
#'     alcohol, drugs, hit and run, roadway departure, rollover, or large
#'     trucks.
#' @param what2 Comparison point for 'what' (set to 'what' unless specified).
#' @param where2 Comparison point for 'where' (set to 'where' unless specified).
#' @param who2 Comparison point for 'who' (set to 'who' unless specified).
#' @param involved2 Comparison point for 'involved' (set to 'involved' unless specified).
#'
#' @return A tibble of counts.
#'
#' @examples
#'
#'   \dontrun{
#'     compare_counts(
#'       get_fars(years = 2020, states="Virginia"),
#'       where = list(urb="rural"),
#'       where2 = list(urb="urban")
#'       )
#'    }




compare_counts <- function(df,
                           interval=c("year", "month")[1],
                           what=c("crashes", "fatalities", "injuries", "people")[1],
                           where=list(states="all",
                                      region = c("all", "ne", "mw", "s", "w")[1],
                                      urb=c("all", "rural", "urban")[1]),
                           who=c("all", "drivers", "passengers", "bicyclists", "pedestrians")[1],
                           involved=NULL,
                           what2=what,
                           where2=where,
                           who2=who,
                           involved2=involved){

  bind_rows(
    counts(df=df, what=what,  where=where,  interval = interval, who = who,  involved = involved),
    counts(df=df, what=what2, where=where2, interval = interval, who = who2, involved = involved2),
    ) %>%
  return()

}
