#' Get Account History For An Account Using Currency
#'
#' @name account_hist
#'
#' @description This is an auth based function. User must have valid api keys generated by GDAX which must be passed as mandatory arguments. The function takes a currency as an additional input and returns the ledger for that currency. Since currency and account id have one to one mapping, the currency is being used a proxy.
#'
#' @param currency Optional character value. The default is \code{"LTC"}. This is case insensitive and must be a valid currency as provided by \code{\link{accounts}} or \code{\link{account}}.
#' @param api.key Mandatory character value. This is the API key as generated by GDAX. Typically a 32 character value.
#' @param secret Mandatory character value. This is the API secret as generated by GDAX. Typically a 88 character value.
#' @param passphrase Mandatory character value. This is the passphrase as generated by GDAX. Typically a 11 character value.
#'
#' @return  Dataframe with account activity for that currency. It indiactes the type of activity, when the activity occured and other associated details.
#'
#' @examples
#' \dontrun{
#' account_hist(api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' account_hist("BTC", api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' account_hist("ETH", api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' }
#'
#' @export

account_hist <- function(currency = "LTC",
                         api.key,
                         secret,
                         passphrase) {
  #Determine account id----
  raw_data <- accounts(api.key, secret, passphrase)

  #case remediation----
  currency <- toupper(currency)

  #valid currency?----
  if (currency %in% raw_data$currency) {
    acct_id <- raw_data$id[raw_data$currency == currency]

    #get url extension----
    req.url <-  paste0("/accounts/", acct_id, "/ledger")

    #define method----
    method <- "GET"

    #fetch response----
    response <- auth(
      method = method,
      req.url = req.url,
      api.key = api.key,
      secret = secret,
      passphrase = passphrase
    )

    #transform----
    response$id <- as.character(response$id)
    response$amount <- as.numeric(response$amount)
    response$balance <- as.numeric(response$balance)

    #return----
    return(response)

  }
  #Invalid currency?----
  else {
    stop("Invalid currency provided")
  }
}
