% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/env.R
\name{get_env}
\alias{get_env}
\alias{set_env}
\title{Get or set the environment of an object}
\usage{
get_env(env = caller_env(), default = NULL)

set_env(env, new_env = caller_env())
}
\arguments{
\item{env}{An environment or an object bundling an environment,
e.g. a formula, \link{quosure} or \link[=is_closure]{closure}.}

\item{default}{The default environment in case \code{env} does not wrap
an environment. If \code{NULL} and no environment could be extracted,
an error is issued.}

\item{new_env}{An environment to replace \code{env} with. Can be an
object handled by \code{get_env()}.}
}
\description{
These functions dispatch internally with methods for functions,
formulas and frames. If called with a missing argument, the
environment of the current evaluation frame (see \code{\link[=ctxt_stack]{ctxt_stack()}}) is
returned. If you call \code{get_env()} with an environment, it acts as
the identity function and the environment is simply returned (this
helps simplifying code when writing generic functions for
environments).
}
\examples{
# Get the environment of frame objects. If no argument is supplied,
# the current frame is used:
fn <- function() {
  list(
    get_env(call_frame()),
    get_env()
  )
}
fn()

# Environment of closure functions:
get_env(fn)

# Or of quosures or formulas:
get_env(~foo)
get_env(quo(foo))


# Provide a default in case the object doesn't bundle an environment.
# Let's create an unevaluated formula:
f <- quote(~foo)

# The following line would fail if run because unevaluated formulas
# don't bundle an environment (they didn't have the chance to
# record one yet):
# get_env(f)

# It is often useful to provide a default when you're writing
# functions accepting formulas as input:
default <- env()
identical(get_env(f, default), default)

# set_env() can be used to set the enclosure of functions and
# formulas. Let's create a function with a particular environment:
env <- child_env("base")
fn <- set_env(function() NULL, env)

# That function now has `env` as enclosure:
identical(get_env(fn), env)
identical(get_env(fn), get_env())

# set_env() does not work by side effect. Setting a new environment
# for fn has no effect on the original function:
other_env <- child_env(NULL)
set_env(fn, other_env)
identical(get_env(fn), other_env)

# Since set_env() returns a new function with a different
# environment, you'll need to reassign the result:
fn <- set_env(fn, other_env)
identical(get_env(fn), other_env)
}
