% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integration.R
\name{runUINMF}
\alias{runUINMF}
\alias{runUINMF.liger}
\title{Perform Mosaic iNMF (UINMF) on scaled datasets with unshared features}
\usage{
runUINMF(object, k = 20, lambda = 5, ...)

\method{runUINMF}{liger}(
  object,
  k = 20,
  lambda = 5,
  nIteration = 30,
  nRandomStarts = 1,
  seed = 1,
  nCores = 2L,
  verbose = getOption("ligerVerbose", TRUE),
  ...
)
}
\arguments{
\item{object}{\linkS4class{liger} object. Should run
\code{\link{selectGenes}} with \code{unshared = TRUE} and then run
\code{\link{scaleNotCenter}} in advance.}

\item{k}{Inner dimension of factorization (number of factors). Generally, a
higher \code{k} will be needed for datasets with more sub-structure. Default
\code{20}.}

\item{lambda}{Regularization parameter. Larger values penalize
dataset-specific effects more strongly (i.e. alignment should increase as
\code{lambda} increases). Default \code{5}.}

\item{...}{Arguments passed to other methods and wrapped functions.}

\item{nIteration}{Total number of block coordinate descent iterations to
perform. Default \code{30}.}

\item{nRandomStarts}{Number of restarts to perform (iNMF objective function
is non-convex, so taking the best objective from multiple successive
initialization is recommended). For easier reproducibility, this increments
the random seed by 1 for each consecutive restart, so future factorization
of the same dataset can be run with one rep if necessary. Default \code{1}.}

\item{seed}{Random seed to allow reproducible results. Default \code{1}.}

\item{nCores}{The number of parallel tasks to speed up the computation.
Default \code{2L}. Only supported for platform with OpenMP support.}

\item{verbose}{Logical. Whether to show information of the progress. Default
\code{getOption("ligerVerbose")} or \code{TRUE} if users have not set.}
}
\value{
\itemize{
 \item{liger method - Returns updated input \linkS4class{liger} object.
 \itemize{
     \item{A list of all \eqn{H} matrices can be accessed with
         \code{getMatrix(object, "H")}}
     \item{A list of all \eqn{V} matrices can be accessed with
         \code{getMatrix(object, "V")}}
     \item{The \eqn{W} matrix can be accessed with
         \code{getMatrix(object, "W")}}
     \item{A list of all \eqn{U} matrices can be accessed with
         \code{getMatrix(object, "U")}}
 }
 }
}
}
\description{
Performs mosaic integrative non-negative matrix factorization (UINMF) (A.R.
Kriebel, 2022) using block coordinate descent (alternating non-negative
least squares, ANLS) to return factorized \eqn{H}, \eqn{W}, \eqn{V} and
\eqn{U} matrices. The objective function is stated as

\deqn{\arg\min_{H\ge0,W\ge0,V\ge0,U\ge0}\sum_{i}^{d}
||\begin{bmatrix}E_i \\ P_i \end{bmatrix} -
(\begin{bmatrix}W \\ 0 \end{bmatrix}+
\begin{bmatrix}V_i \\ U_i \end{bmatrix})Hi||^2_F+
\lambda_i\sum_{i}^{d}||\begin{bmatrix}V_i \\ U_i \end{bmatrix}H_i||_F^2}

where \eqn{E_i} is the input non-negative matrix of the \eqn{i}'th dataset,
\eqn{P_i} is the input non-negative matrix for the unshared features,
\eqn{d} is the total number of datasets. \eqn{E_i} is of size
\eqn{m \times n_i} for \eqn{m} shared features and \eqn{n_i} cells, \eqn{P_i}
is of size \eqn{u_i \times n_i} for \eqn{u_i} unshared feaetures,
\eqn{H_i} is of size \eqn{k \times n_i}, \eqn{V_i} is of size
\eqn{m \times k}, \eqn{W} is of size \eqn{m \times k} and \eqn{U_i} is of
size \eqn{u_i \times k}.

The factorization produces a shared \eqn{W} matrix (genes by k). For each
dataset, an \eqn{H} matrix (k by cells), a \eqn{V} matrix (genes by k) and
a \eqn{U} matrix (unshared genes by k). The \eqn{H} matrices represent the
cell factor loadings. \eqn{W} is held consistent among all datasets, as it
represents the shared components of the metagenes across datasets. The
\eqn{V} matrices represent the dataset-specific components of the metagenes,
\eqn{U} matrices are similar to \eqn{V}s but represents the loading
contributed by unshared features.

This function adopts highly optimized fast and memory efficient
implementation extended from Planc (Kannan, 2016). Pre-installation of
extension package \code{RcppPlanc} is required. The underlying algorithm
adopts the identical ANLS strategy as \code{\link{optimizeALS}(unshared =
TRUE)} in the old version of LIGER.
}
\note{
Currently, Seurat S3 method is not supported for UINMF because there is no
simple solution for organizing a number of miscellaneous matrices with a
single Seurat object. We strongly recommend that users create a
\linkS4class{liger} object which has the specific structure.
}
\examples{
pbmc <- normalize(pbmc)
pbmc <- selectGenes(pbmc, useUnsharedDatasets = c("ctrl", "stim"))
pbmc <- scaleNotCenter(pbmc)
if (!is.null(getMatrix(pbmc, "scaleUnsharedData", "ctrl")) &&
    !is.null(getMatrix(pbmc, "scaleUnsharedData", "stim"))) {
    # TODO: unshared variable features cannot be detected from this example
    pbmc <- runUINMF(pbmc)
}
}
\references{
April R. Kriebel and Joshua D. Welch, UINMF performs mosaic
integration of single-cell multi-omic datasets using nonnegative matrix
factorization, Nat. Comm., 2022
}
