#' Make tabular summary of a monadic chain
#'
#' @family monad-to-x
#' @param m An Rmonad
#' @export
mtabulate <- function(m){
  ms <- as.list(m)
  d <- do.call(rbind.data.frame, lapply(ms, .mtabulate)) %>%
    as.data.frame
  d
}
.mtabulate <- function(m){
  list(
    code      = paste(m_code(m), collapse="\n"),
    OK        = m_OK(m),
    cached    = !is.null(m_value(m)),
    time      = signif(m_time(m)[1], 2),
    space     = m_mem(m),
    nbranch   = length(m_branch(m)),
    nnotes    = length(m_notes(m)),
    nwarnings = length(m_warnings(m)),
    error     = length(m_error(m)),
    doc       = length(m_doc(m))
  )
}

#' Tabulates all errors, warnings and notes
#' 
#' @family monad-to-x
#' @param m An Rmonad
#' @export
missues <- function(m){
  ms <- as.list(m)
  cid <- 1L
  .missues <- function(m) {
    type <- c(
              rep.int("error",   length(m_error(m))    ),
              rep.int("warning", length(m_warnings(m)) ),
              rep.int("note",    length(m_notes(m))    )
             )
    issue <- as.character(c(m_error(m), m_warnings(m), m_notes(m)))
    idcol <- rep(cid, length(type))
    cid <<- cid + 1
    list(id=idcol, type=type, issue=issue) 
  }
  do.call(rbind.data.frame, lapply(ms, .missues)) %>%
    as.data.frame  # NOTE: this cast is required, since the above code
                   # silently mishandles the case or a zero-row data
                   # frame (it returns a list).
}

#' Returns the value of a monad holds
#'
#' If the monad is in the passing state, return the wrapped value. Otherwise,
#' raise an appropriate error.
#'
#' Regardless of pass/fail status, \code{esc} raises all collected warnings and
#' prints all messages. Terminating a monadic sequence with \code{esc} should
#' obtain a result very close to running the same code outside the monad. The
#' main difference is that Rmonad appends the toplevel code that generated the
#' error.
#'
#' @family monad-to-x
#' @param m An Rmonad
#' @export 
esc <- function(m){
  issues <- merge(mtabulate(m), missues(m))[, c("id", "code", "type", "issue")]
  for(i in seq_len(nrow(issues))){
    # raise warnings, with contextual information
    if(issues[i, "type"] == "warning"){
      warning("in '", issues[i, "code"], "': ", issues[i, "issue"], call.=FALSE)
    }
    # pass messages verbatim
    if(issues[i, "type"] == "note"){
      message(issues[i, "issue"])
    }
  }
  if(! m_OK(m)){
    # if the final state is failing, raise error with contextual info
    msg <- paste0('in "', m_code(m), '":\n  ', m_error(m))
    stop(msg, call.=FALSE)
  }
  m_value(m)
}

#' Return each independent branch of the pipeline
#'
#' Branches are generated by the `%>^%` operator. The allow a value to be sent
#' into a new branch of the pipeline, which can fail independently, without
#' propagating to other pieces of the program.
#'
#' @param m An Rmonad
#' @return A list of Rmonads, one for each branch
#' @export
unbranch <- function(m){

  as.list(m)               %>%
    Filter(f=.has_branch)  %>%
    lapply(m_branch)       %>%
    lapply(rev)            %>%
    unlist                 %>%
    append(x=list(m))

}
