# --------------------------------------
# Author: Andreas Alfons
#         Erasmus Universiteit Rotterdam
# --------------------------------------

#' (Robust) mediation analysis
#'
#' Perform (robust) mediation analysis via a bootstrap test or Sobel's test.
#'
#' If \code{method} is \code{"covariance"} and \code{robust} is \code{TRUE},
#' the tests are based on a Huber M-estimator of location and scatter.  For the
#' bootstrap test, the M-estimates are used to first clean the data via a
#' transformation.  Then the standard bootstrap is performed with the cleaned
#' data.  Note that this does not account for the variability from cleaning the
#' data.
#'
#' A much more robust method based on robust regression will be available
#' soon.  Currently, least squares regression is always performed if
#' \code{method} is \code{"regression"}.
#'
#' \code{indirect} is a wrapper function for performing non-robust mediation
#' analysis via regressions (inspired by Preacher & Hayes' \code{SPSS} macro
#' \code{INDIRECT}).
#'
#' @aliases print.bootTestMediation print.sobelTestMediation
#' summary.testMediation
#'
#' @param x  either a numeric vector containing the independent variable, or
#' (if \code{data} is supplied) a character string, an integer or a logical
#' vector specifying the corresponding column of \code{data}.
#' @param y  either a numeric vector containing the dependent variable, or
#' (if \code{data} is supplied) a character string, an integer or a logical
#' vector specifying the corresponding column of \code{data}.
#' @param m  either a numeric vector containing the proposed mediator variable,
#' or (if \code{data} is supplied) a character string, an integer or a logical
#' vector specifying the corresponding column of \code{data}.
#' @param covariates  optional; either a numeric vector or data frame
#' containing additional covariates to be used as control variables, or (if
#' \code{data} is supplied) a character, integer or logical vector specifying
#' the corresponding column of \code{data}.
#' @param data  an optional \code{data.frame}.
#' @param test  a character string specifying the test to be performed for
#' the indirect effect.  Possible values are \code{"boot"} (the default) for
#' the bootstrap, or \code{"sobel"} for Sobel's test.
#' @param alternative  a character string specifying the alternative hypothesis
#' in the test for the indirect effect.  Possible values are \code{"twosided"}
#' (the default), \code{"less"} or \code{"greater"}.
#' @param R  an integer giving the number of bootstrap replicates.  The default
#' is to use 5000 bootstrap replicates.
#' @param level  numeric; the confidence level of the confidence interval in
#' the bootstrap test.  The default is to compute a 95\% confidence interval.
#' @param type  a character string specifying the type of confidence interval
#' to be computed in the bootstrap test.  Possible values are \code{"bca"} (the
#' default) for the bias-corrected and accelerated bootstrap, or \code{"perc"}
#' for the percentile bootstrap.
#' @param method  a character string specifying the method of estimation for
#' the mediation model.  Possible values are \code{"covariance"} (the default)
#' to estimate the effects via the covariance matrix, or \code{"regression"}
#' to estimate the effects via regressions.  Note that the effects are always
#' estimated via regressions if control variables are specified via
#' \code{covariates}.
#' @param robust  a logical indicating whether to perform a robust test
#' (defaults to \code{TRUE}).  This is currently ignored if \code{method} is
#' \code{"regression"}.
#' @param control  if \code{robust} is \code{TRUE} and \code{method} is
#' \code{"covariance"}, a list of tuning parameters for \code{\link{covHuber}}
#' as generated by \code{\link{covControl}}.
#' @param \dots  additional arguments to be passed to \code{\link[boot]{boot}}
#' in case of a bootstrap test.
#'
#' @return An object inheriting from class \code{"testMediation"} (class
#' \code{"bootTestMediation"} if \code{test} is \code{"boot"} or
#' \code{"sobelTestMediation"} if \code{test} is \code{"sobel"}) with the
#' following components:
#' \item{ab}{numeric; the point estimate of the indirect effect.}
#' \item{ci}{a numeric vector of length two containing the bootstrap
#' confidence interval for the indirect effect (only
#' \code{"bootTestMediation"}).}
#' \item{reps}{an object of class \code{"\link[boot]{boot}"} containing
#' the bootstrap replicates of the indirect effect (only
#' \code{"bootTestMediation"}).}
#' \item{se}{numeric; the standard error of the indirect effect according
#' to Sobel's formula (only \code{"sobelTestMediation"}).}
#' \item{statistic}{numeric; the test statistic for Sobel's test (only
#' \code{"sobelTestMediation"}).}
#' \item{pValue}{numeric; the p-Value from Sobel's test (only
#' \code{"sobelTestMediation"}).}
#' \item{alternative}{a character string specifying the alternative
#' hypothesis in the test for the indirect effect.}
#' \item{R}{an integer giving the number of bootstrap replicates (only
#' \code{"bootTestMediation"}).}
#' \item{level}{numeric; the confidence level of the bootstrap confidence
#' interval (only \code{"bootTestMediation"}).}
#' \item{type}{a character string specifying the type of bootstrap
#' confidence interval (only \code{"bootTestMediation"}).}
#' \item{fit}{an object inheriting from class
#' \code{"\link{fitMediation}"} containing the estimation results for the
#' direct effect and the total effect in the mediation model.}
#'
#' @author Andreas Alfons
#'
#' @references
#' Preacher, K.J. and Hayes, A.F. (2004) SPSS and SAS procedures for estimating
#' indirect effects in simple mediation models. \emph{Behavior Research Methods,
#' Instruments, & Computers}, \bold{36}(4), 717--731.
#'
#' Sobel, M.E. (1982) Asymptotic confidence intervals for indirect effects in
#' structural equation models. \emph{Sociological Methodology}, \bold{13},
#' 290--312.
#'
#' Zu, J. and Yuan, K.-H. (2010) Local influence and robust procedures for
#' mediation analysis. \emph{Multivariate Behavioral Research}, \bold{45}(1),
#' 1--44.
#'
#' @seealso \code{\link{fitMediation}}
#'
#' \code{\link[=coef.testMediation]{coef}},
#' \code{\link[=confint.testMediation]{confint}},
#' \code{\link[=fortify.testMediation]{fortify}} and
#' \code{\link[=plotMediation]{plot}} methods
#'
#' \code{\link[boot]{boot}}, \code{\link{covHuber}}, \code{\link{covML}},
#' \code{\link[stats]{lm}}
#'
#' @examples
#' # control parameters
#' n <- 250             # number of observations
#' a <- b <- c <- 0.2   # true effects
#' t <- 2               # number of observations to contaminate
#'
#' # draw clean observations
#' set.seed(20160911)
#' x <- rnorm(n)
#' m <- a * x + rnorm(n)
#' y <- b * m + c * x + rnorm(n)
#'
#' # contaminate the first t observations
#' m[1:t] <- m[1:t] - 6
#' y[1:t] <- y[1:t] + 6
#'
#' # perform mediation analysis
#' test <- testMediation(x, y, m)
#' summary(test)
#'
#' @keywords multivariate
#'
#' @import boot
#' @export

testMediation <- function(x, ...) UseMethod("testMediation")


#' @rdname testMediation
#' @method testMediation default
#' @export

testMediation.default <- function(x, y, m, covariates = NULL, data,
                                  test = c("boot", "sobel"),
                                  alternative = c("twosided", "less", "greater"),
                                  R = 5000, level = 0.95,
                                  type = c("bca", "perc"),
                                  method = c("covariance", "regression"),
                                  robust = TRUE, control, ...) {
  ## fit mediation model
  fit <- fitMediation(x, y, m, covariates=covariates, data=data, method=method,
                      robust=robust, control=control)
  ## call method for fitted model
  testMediation(fit, test=test, alternative=alternative,
                R=R, level=level, type=type, ...)
}


#' @rdname testMediation
#' @method testMediation fitMediation
#' @export

testMediation.fitMediation <- function(x, test = c("boot", "sobel"),
                                       alternative = c("twosided", "less", "greater"),
                                       R = 5000, level = 0.95,
                                       type = c("bca", "perc"),
                                       ...) {
  ## initializations
  test <- match.arg(test)
  alternative <- match.arg(alternative)
  ## perform mediation analysis
  if(test == "boot") {
    # further inizializations
    level <- rep(as.numeric(level), length.out=1)
    if(is.na(level) || level < 0 || level > 1) level <- formals()$level
    type <- match.arg(type)
    # perform bootstrap test
    bootTestMediation(x, alternative=alternative, R=R,
                      level=level, type=type, ...)
  } else if(test == "sobel") {
    # perform Sobel test
    sobelTestMediation(x, alternative=alternative)
  } else stop("test not implemented")
}


#' @rdname testMediation
#' @export

indirect <- function(..., test = c("boot", "sobel"), method = "regression",
                     robust = FALSE) {
  testMediation(..., test=test, method="regression", robust=FALSE)
}


## internal function for bootstrap test
bootTestMediation <- function(fit,
                              alternative = c("twosided", "less", "greater"),
                              R = 5000, level = 0.95, type = c("bca", "perc"),
                              ...) {
  if(inherits(fit, "regFitMediation")) {
    # indices of covariates in data matrix to be used in bootstrap
    j <- match(fit$covariates, names(fit$data)) + 1
    n <- nrow(fit$data)
    # check if fast and robust bootstrap should be applied
    # combine data
    z <- cbind(rep.int(1, n), as.matrix(fit$data))
    # perform bootstrap
    bootstrap <- localBoot(z, function(z, i) {
      # extract bootstrap sample from the data
      zi <- z[i, , drop=FALSE]
      # compute coefficients from regression m ~ x + covariates
      xi <- zi[, c(1:2, j)]
      mi <- zi[, 4]
      coefMi <- drop(solve(crossprod(xi)) %*% crossprod(xi, mi))
      # compute coefficients from regression y ~ m + x + covariates
      mxi <- zi[, c(1, 4, 2, j)]
      yi <- zi[, 3]
      coefYi <- drop(solve(crossprod(mxi)) %*% crossprod(mxi, yi))
      # compute effects
      unname(coefMi[2]) * unname(coefYi[2])
      a <- unname(coefMi[2])
      b <- unname(coefYi[2])
      c <- unname(coefYi[3])
      ab <- a * b
      cPrime <- ab + c
      # compute effects of control variables if they exist
      covariates <- unname(coefYi[-(1:3)])
      # return effects
      c(ab, a, b, c, cPrime, covariates)
    }, R=R, ...)
    R <- nrow(bootstrap$t)  # make sure that number of replicates is correct
  } else if(inherits(fit, "covFitMediation")) {
    # extract data and variable names
    x <- fit$x
    y <- fit$y
    m <- fit$m
    data <- fit$data
    # check if the robust transformation of Zu & Yuan (2010) should be applied
    if(fit$robust) {
      cov <- fit$cov
      data[] <- mapply("-", data, cov$center, SIMPLIFY=FALSE, USE.NAMES=FALSE)
      data <- weights(cov, type="consistent") * data
    }
    # perform bootstrap
    bootstrap <- localBoot(data, function(z, i) {
      # extract bootstrap sample from the data
      zi <- z[i, , drop=FALSE]
      # compute MLE of covariance matrix on bootstrap sample
      S <- covML(zi)$cov
      # compute effects
      a <- S[m, x] / S[x, x]
      det <- S[x, x] * S[m, m] - S[m, x]^2
      b <- (-S[m, x] * S[y, x] + S[x, x] * S[y, m]) / det
      c <- (S[m, m] * S[y, x] - S[m, x] * S[y, m]) / det
      cPrime <- S[y, x] / S[x, x]
      c(a*b, a, b, c, cPrime)
    }, R=R, ...)
    R <- nrow(bootstrap$t)  # make sure that number of replicates is correct
  } else stop("method not implemented")
  # extract confidence interval for indirect effect
  ci <- confint(bootstrap, parm=1, level=level, alternative=alternative,
                type=type)
  # construct return object
  result <- list(ab=mean(bootstrap$t[, 1], na.rm=TRUE), ci=ci, reps=bootstrap,
                 alternative=alternative, R=R[1], level=level, type=type,
                 fit=fit)
  class(result) <- c("bootTestMediation", "testMediation")
  result
}

## internal function for sobel test
sobelTestMediation <- function(fit,
                               alternative = c("twosided", "less", "greater"),
                               ...) {
  # extract coefficients
  a <- fit$a
  b <- fit$b
  # compute standard errors
  summary <- getSummary(fit)
  sa <- summary$a[1, 2]
  sb <- summary$b[1, 2]
  # compute test statistic and p-Value
  ab <- a * b
  se <- sqrt(b^2 * sa^2 + a^2 * sb^2)
  z <- ab / se
  pValue <- pValueZ(z, alternative=alternative)
  # construct return item
  result <- list(ab=ab, se=se, statistic=z, pValue=pValue,
                 alternative=alternative, fit=fit)
  class(result) <- c("sobelTestMediation", "testMediation")
  result
}

## wrapper function for boot() that ignores unused arguments, but allows
## arguments for parallel computing to be passed down
localBoot <- function(..., sim, stype, L, m, ran.gen, mle) boot(...)

## internal function to compute p-value based on normal distribution
pValueZ <- function(z, alternative = c("twosided", "less", "greater")) {
  # initializations
  alternative <- match.arg(alternative)
  # compute p-value
  switch(alternative, twosided=2*pnorm(abs(z), lower.tail=FALSE),
         less=pnorm(z), greater=pnorm(z, lower.tail=FALSE))
}
