% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brsmatch.R
\name{brsmatch}
\alias{brsmatch}
\title{Balanced Risk Set Matching}
\usage{
brsmatch(
  n_pairs,
  data,
  id = "id",
  time = "time",
  trt_time = "trt_time",
  covariates = NULL,
  balance = TRUE,
  balance_covariates = NULL,
  exact_match = NULL,
  options = list(time_lag = FALSE, verbose = FALSE, optimizer = c("glpk", "gurobi"))
)
}
\arguments{
\item{n_pairs}{The number of pairs desired from matching.}

\item{data}{A data.frame or similar containing columns matching the \verb{id, time, trt_time} arguments, and covariates. This data frame is expected to
be in tidy, long format, so that \code{id}, \code{trt_time}, and other variables may
be repeated for different values of \code{time}. The data.frame should be unique
at \code{id} and \code{time}.}

\item{id}{A character specifying the id column name (default \code{'id'}).}

\item{time}{A character specifying the time column name (default \code{'time'}).}

\item{trt_time}{A character specifying the treatment time column name
(default \code{'trt_time'}).}

\item{covariates}{A character vector specifying the covariates to use for
matching (default \code{NULL}). If \code{NULL}, this will default to all columns
except those named by the \code{id}, \code{time}, and \code{trt_time} arguments.}

\item{balance}{A logical value indicating whether to include balancing
constraints in the matching process.}

\item{balance_covariates}{A character vector specifying the covariates to use
for balancing (default \code{NULL}).  If \code{NULL}, this will default to all
columns except those named by the \code{id}, \code{time}, and \code{trt_time} arguments.}

\item{exact_match}{A vector of optional covariates to perform exact matching
on. If \code{NULL}, no exact matching is done.}

\item{options}{A list of additional parameters with the following components:
\itemize{
\item \code{time_lag} A logical value indicating whether the matches should be made
on the time period preceding treatment.  This can help avoid confounding if
treatment happens between two periods.
\item \code{verbose} A logical value
indicating whether to print information to the console during a potentially
long matching process.
\item \code{optimizer} The optimizer to use (default
\code{'glpk'}). The option \code{'gurobi'} requires an external license and package,
but offers speed improvements.
}}
}
\value{
A data frame containing the pair information.  The data frame has
columns \code{id}, \code{pair_id}, and \code{type}. \code{id} matches the input parameter and
will contain all ids from the input data frame.  \code{pair_id} refers to the id
of the computed pairs; \code{NA} values indicate unmatched individuals.  \code{type}
indicates whether the individual in the pair is considered as treatment
("trt") or control ("all") in that pair.
}
\description{
Perform balanced risk set matching as described in Li et al. (2001) "Balanced
Risk Set Matching".  Given a longitudinal data frame with covariate
information, along with treatment time, build a MIP problem that matches
treated individuals to those that haven't been treated yet (or are never
treated) based on minimizing the Mahalanobis distance between covariates. If
balancing is desired, the model will try to minimize the imbalance in terms
of specified balancing covariates in the final pair output.  Each treated
individual is matched to one other individual.
}
\details{
Note that when using exact matching, the \code{n_pairs} are split roughly in
proportion to the number of treated subjects in each exact matching group.
This has a possibility of failing  when \code{n_pairs} is large.  If this happens
to you, we suggest manually performing exact matching, for example with
\code{split()}, and selecting \code{n_pairs} for each group interactively.
}
\examples{
if (requireNamespace("Rglpk", quietly = TRUE)) {
  library(dplyr, quietly = TRUE)
  pairs <- brsmatch(
    n_pairs = 13,
    data = oasis,
    id = "subject_id",
    time = "visit",
    trt_time = "time_of_ad",
    balance = FALSE
  )

  na.omit(pairs)

  # evaluate the first match
  first_match <- pairs$subject_id[which(pairs$pair_id == 1)]
  oasis \%>\% dplyr::filter(subject_id \%in\% first_match)
}

}
\references{
Li, Yunfei Paul, Kathleen J Propert, and Paul R Rosenbaum. 2001.
"Balanced Risk Set Matching." Journal of the American Statistical
Association 96 (455): 870-82.
\doi{10.1198/016214501753208573}
}
\author{
Sean Kent
}
