% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pp_check.R
\name{pp_check.stanreg}
\alias{pp_check.stanreg}
\alias{pp_check}
\title{Graphical posterior predictive checks}
\usage{
\method{pp_check}{stanreg}(object, plotfun = "dens_overlay", nreps = NULL, seed = NULL, ...)
}
\arguments{
\item{object}{A fitted model object returned by one of the 
\pkg{rstanarm} modeling functions. See \code{\link{stanreg-objects}}.}

\item{plotfun}{A character string naming the \pkg{bayesplot} 
\link[bayesplot:PPC-overview]{PPC} function to use. The default is to call
\code{\link[bayesplot:PPC-distributions]{ppc_dens_overlay}}. \code{plotfun} can be specified
either as the full name of a \pkg{bayesplot} plotting function (e.g.
\code{"ppc_hist"}) or can be abbreviated to the part of the name following
the \code{"ppc_"} prefix (e.g. \code{"hist"}). To get the names of all
available PPC functions see \code{\link[bayesplot]{available_ppc}}.}

\item{nreps}{The number of \eqn{y^{rep}}{yrep} datasets to generate from the 
\link[=posterior_predict]{posterior predictive distribution} and show in
the plots. The default depends on \code{plotfun}. For functions that plot
each \code{yrep} dataset separately (e.g. \code{ppc_hist}), \code{nreps}
defaults to a small value to make the plots readable. For functions that
overlay many \code{yrep} datasets (e.g., \code{ppc_dens_overlay}) a larger
number is used by default, and for other functions (e.g. \code{ppc_stat})
the default is to set \code{nreps} equal to the posterior sample size.}

\item{seed}{An optional \code{\link[=set.seed]{seed}} to pass to 
\code{\link{posterior_predict}}.}

\item{...}{Additonal arguments passed to the \pkg{\link{bayesplot}} function 
called. For many plotting functions \code{...} is optional, however for 
functions that require a \code{group} or \code{x} argument, these arguments
should be specified in \code{...}. If specifying \code{group} and/or 
\code{x}, they can be provided as either strings naming variables (in which
case they are searched for in the model frame) or as vectors containing the
actual values of the variables. See the \strong{Examples} section, below.}
}
\value{
\code{pp_check} returns a ggplot object that can be further
  customized using the \pkg{ggplot2} package.
}
\description{
Interface to the \link[bayesplot:PPC-overview]{PPC} (posterior predictive checking) module
in the \pkg{\link{bayesplot}} package, providing various plots comparing the 
observed outcome variable \eqn{y} to simulated datasets \eqn{y^{rep}}{yrep} 
from the posterior predictive distribution. The \code{pp_check} method for 
\link{stanreg-objects} prepares the arguments required for the specified 
\pkg{bayesplot} PPC plotting function and then calls that function. It is 
also straightforward to use the functions from the \pkg{bayesplot} package 
directly rather than via the \code{pp_check} method. Examples of both are
given below.
}
\note{
For binomial data, plots of \eqn{y} and \eqn{y^{rep}}{yrep} show the 
  proportion of 'successes' rather than the raw count. Also for binomial 
  models see \code{\link[bayesplot:PPC-errors]{ppc_error_binned}} for binned residual
  plots.
}
\examples{
if (.Platform$OS.type != "windows" || .Platform$r_arch != "i386") {
fit <- stan_glmer(
  mpg ~ wt + am + (1|cyl), 
  data = mtcars, 
  iter = 400, # iter and chains small just to keep example quick
  chains = 2, 
  refresh = 0
) 

# Compare distribution of y to distributions of multiple yrep datasets
pp_check(fit)
pp_check(fit, plotfun = "boxplot", nreps = 10, notch = FALSE)
pp_check(fit, plotfun = "hist", nreps = 3)

\donttest{
# Same plot (up to RNG noise) using bayesplot package directly
bayesplot::ppc_hist(y = mtcars$mpg, yrep = posterior_predict(fit, draws = 3))

# Check histograms of test statistics by level of grouping variable 'cyl'
pp_check(fit, plotfun = "stat_grouped", stat = "median", group = "cyl")

# Defining a custom test statistic
q25 <- function(y) quantile(y, probs = 0.25) 
pp_check(fit, plotfun = "stat_grouped", stat = "q25", group = "cyl")

# Scatterplot of two test statistics
pp_check(fit, plotfun = "stat_2d", stat = c("mean", "sd"))

# Scatterplot of y vs. average yrep
pp_check(fit, plotfun = "scatter_avg") # y vs. average yrep
# Same plot (up to RNG noise) using bayesplot package directly
bayesplot::ppc_scatter_avg(y = mtcars$mpg, yrep = posterior_predict(fit))

# Scatterplots of y vs. several individual yrep datasets
pp_check(fit, plotfun = "scatter", nreps = 3)

# Same plot (up to RNG noise) using bayesplot package directly
bayesplot::ppc_scatter(y = mtcars$mpg, yrep = posterior_predict(fit, draws = 3))

# yrep intervals with y points overlaid
# by default 1:length(y) used on x-axis but can also specify an x variable
pp_check(fit, plotfun = "intervals")
pp_check(fit, plotfun = "intervals", x = "wt") + ggplot2::xlab("wt")

# Same plot (up to RNG noise) using bayesplot package directly
bayesplot::ppc_intervals(y = mtcars$mpg, yrep = posterior_predict(fit), 
                         x = mtcars$wt) + ggplot2::xlab("wt")

# predictive errors
pp_check(fit, plotfun = "error_hist", nreps = 6)
pp_check(fit, plotfun = "error_scatter_avg_vs_x", x = "wt") + 
  ggplot2::xlab("wt")
  
# Example of a PPC for ordinal models (stan_polr)
fit2 <- stan_polr(tobgp ~ agegp, data = esoph, method = "probit",
                  prior = R2(0.2, "mean"), init_r = 0.1, 
                  refresh = 0)
pp_check(fit2, plotfun = "bars", nreps = 500, prob = 0.5)
pp_check(fit2, plotfun = "bars_grouped", group = esoph$agegp, 
         nreps = 500, prob = 0.5)
}
}
}
\references{
Gelman, A., Carlin, J. B., Stern, H. S., Dunson, D. B., Vehtari,
  A., and Rubin, D. B. (2013). \emph{Bayesian Data Analysis.} Chapman & Hall/CRC
  Press, London, third edition. (Ch. 6)

Gabry, J. , Simpson, D. , Vehtari, A. , Betancourt, M. and
  Gelman, A. (2019), Visualization in Bayesian workflow. \emph{J. R. Stat.
  Soc. A}, 182: 389-402. doi:10.1111/rssa.12378,
  \href{https://arxiv.org/abs/1709.01449}{arXiv preprint},
  \href{https://github.com/jgabry/bayes-vis-paper}{code on GitHub})
}
\seealso{
\itemize{
  \item The vignettes in the \pkg{bayesplot} package for many examples.
    Examples of posterior predictive checks can also be found in the
    \pkg{rstanarm} vignettes and demos.
  \item \code{\link[bayesplot]{PPC-overview}} (\pkg{bayesplot}) for links to 
    the documentation for all the available plotting functions.
  \item \code{\link{posterior_predict}} for drawing from the posterior 
    predictive distribution. 
  \item \code{\link[bayesplot:bayesplot-colors]{color_scheme_set}} to change the color scheme 
    of the plots.
}
}
