% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rpca.R
\name{rpca}
\alias{rpca}
\title{Randomized principal component analysis (rpca).}
\usage{
rpca(A, k = NULL, center = TRUE, scale = TRUE, loading = FALSE,
  retx = FALSE, svdalg = "auto", p = 10, q = 1, ...)
}
\arguments{
\item{A}{array_like \cr
a numeric input matrix (or data frame), with dimensions \eqn{(m, n)}. \cr
If the data contain \eqn{NA}s na.omit is applied.}

\item{k}{int, optional \cr
determines the number of principle components to compute. It is required that \eqn{k} is smaller or equal to
\eqn{n}, but it is recommended that \eqn{k << min(m,n)}.}

\item{center}{bool (\eqn{TRUE}, \eqn{FALSE}), optional \cr
a logical value (\eqn{TRUE} by default) indicating whether the variables should be
shifted to be zero centered. Alternatively, a vector of length equal the number of
columns of \eqn{A} can be supplied. The value is passed to scale.}

\item{scale}{bool (\eqn{TRUE}, \eqn{FALSE}), optional \cr
a logical value (\eqn{TRUE} by default) indicating whether the variables should
be scaled to have unit variance. Alternatively, a vector of length equal the number of
columns of \eqn{A} can be supplied. The value is passed to scale.}

\item{loading}{bool (\eqn{TRUE}, \eqn{FALSE}), optional \cr
When \eqn{TRUE} (by default \eqn{FALSE}) the eigenvectors
are unit scaled by the square root of the eigenvalues \eqn{W = W * diag(sqrt(eigvals))}.}

\item{retx}{bool (\eqn{TRUE}, \eqn{FALSE}), optional \cr
a logical value (\eqn{FALSE} by default) indicating whether the rotated variables / scores
should be returned.}

\item{svdalg}{str c('auto', 'reigen', 'rsvd', svd'), optional \cr
Determines which algorithm should be used for computing the singular value decomposition.
By default 'auto' is used, which decides whether to use \code{\link{reigen}} or \code{\link{svd}},
depending on the number of principle components. If \eqn{k < min(n,m)/1.5} randomized reigen is used.
Instead \code{\link{rsvd}} can be used, as well.}

\item{p}{int, optional \cr
oversampling parameter for \eqn{reigen}  (default \eqn{p=10}), see \code{\link{reigen}}.}

\item{q}{int, optional \cr
number of power iterations  for \eqn{reigen} (default \eqn{q=1}), see \code{\link{reigen}}.}

\item{...}{arguments passed to or from other methods, see \code{\link{reigen}}.}

\item{.................}{.}
}
\value{
\code{rpca} returns a list with class \eqn{rpca} containing the following components:
   \item{rotation}{  array_like \cr
                     matrix containing the rotation (eigenvectors),
                     or the variable loadings if \eqn{loadings=TRUE}; array with dimensions \eqn{(n, k)}.
   }
   \item{loading}{  array_like \cr
                     matrix containing the loadings (scaled eigenvectors),
                     if \eqn{loadings=TRUE}; array with dimensions \eqn{(n, k)}.
   }

   \item{eigvals}{  array_like \cr
                    the eigenvalues; 1-d array of length \eqn{k}.
   }
   \item{sdev}{     array_like \cr
                    the standard deviations of the principal components.
   }
   \item{x}{        array_like \cr
                    if \eqn{retx} is true a matrix containing the scores / rotated data
                    (centred and scaled if requested) is returned.
   }
   \item{center, scale}{  array_like \cr
                    the centering and scaling used, or \eqn{FALSE}.
   }
   \item{.................}{.}
}
\description{
Principal components analysis using randomized singular value decomposition.
}
\details{
Principal component analysis is a linear dimensionality reduction technique,
aiming to keep only the most significant principal components to allow
a better interpretation of the data and to project the data to a lower dimensional space.

Traditionally, the computation is done by a (deterministic) singular value decomposition.
Randomized PCA is computed using a fast randomized algorithm (\code{\link{rsvd}})
to compute the approximate low-rank SVD decomposition.
The computational gain is high if the desired number of principal components
is small, i.e. \eqn{k << n}.

\code{\link{rsvd}} expects a numeric (real/complex) input matrix with dimensions \eqn{(m, n)}.
Given a target rank \eqn{k}, \code{rsvd} factors the input matrix \eqn{A} as
\eqn{A = W * diag(s) * W'}. The columns of the real or complex unitary matrix \eqn{W}
contain the eigenvectors (i.e. principal components). The vector \eqn{s} contains the corresponding
eigenvalues. Following \code{\link{prcomp}} we denote this matrix \eqn{W} as
rotation matrix (commonly also called loadings).

The print and summary method can be used to present the results in a nice format.
A scree plot can be produced with the plot function or as recommended with
\code{\link{ggscreeplot}}. A biplot can be produced with \code{\link{ggbiplot}},
and a correlation plot with \code{\link{ggcorplot}}.

The predict function can be used to compute the scores of new observations. The data
will automatically be centred (and scaled if requested). This is not fully supported for
complex input matrices.
}
\note{
The principal components are not unique and only defined up to sign
(a constant of modulus one in the complex case) and so may differ between different
 PCA implementations.

Similar to \code{\link{prcomp}} the variances are computed with the usual divisor N - 1.

Note also that \eqn{scale = TRUE} cannot be used if there are zero or constant (for \eqn{center = TRUE} ) variables.
}
\examples{

library(rsvd)
#
# Load Edgar Anderson's Iris Data
#
data(iris)

#
# log transform
#
log.iris <- log( iris[ , 1:4] )
iris.species <- iris[ , 5]

#
# Perform rPCA and compute all PCs, similar to \\code{\\link{prcomp}}
#
iris.rpca <- rpca(log.iris, retx=TRUE)
summary(iris.rpca) # Summary
print(iris.rpca) # Prints the rotations

# You can compare the results with prcomp
# iris.pca <- prcomp(log.iris, center = TRUE, scale. = TRUE)
# summary(iris.pca) # Summary
# print(iris.pca) # Prints the rotations

#
# Plot functions
#
ggscreeplot(iris.rpca) # Screeplot (explained variance/eigenvalues)
ggscreeplot(iris.rpca, 'ratio') # Proportion of variance
ggscreeplot(iris.rpca, 'cum') # Cummulative proportion

ggcorplot(iris.rpca, pcs=c(1,2)) # The correlation of the original variable with the PCs

ggbiplot(iris.rpca, groups = iris.species, circle = FALSE) #Biplot

#
# Perform rPCA and compute only the first two PCs
#
iris.rpca <- rpca(log.iris, k=2,  svdalg = 'rsvd')
summary(iris.rpca) # Summary
print(iris.rpca) # Prints the rotations

#
# Compute the scores of new observations
#
preds <- predict(iris.rpca, newdata=data.frame(log.iris))

}
\author{
N. Benjamin Erichson, \email{nbe@st-andrews.ac.uk}
}
\seealso{
\code{\link{ggscreeplot}}, \code{\link{ggbiplot}},
         \code{\link{ggcorplot}}, \code{\link{plot.rpca}},
         \code{\link{predict}},   \code{\link{rsvd}}
}

