% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rpca.R
\name{rpca}
\alias{rpca}
\title{Randomized principal component analysis (rpca).}
\usage{
rpca(A, k = NULL, center = TRUE, scale = TRUE, retx = TRUE,
  p = 10, q = 2, rand = TRUE)
}
\arguments{
\item{A}{array_like; \cr
a numeric \eqn{(m, n)} input matrix (or data frame) to be analyzed. \cr
If the data contain \eqn{NA}s na.omit is applied.}

\item{k}{integer; \cr
number of dominant principle components to be computed. It is required that \eqn{k} is smaller or equal to
\eqn{min(m,n)}, but it is recommended that \eqn{k << min(m,n)}.}

\item{center}{bool, optional; \cr
logical value which indicates whether the variables should be
shifted to be zero centered (\eqn{TRUE} by default).}

\item{scale}{bool, optional; \cr
logical value which indicates whether the variables should
be scaled to have unit variance (\eqn{TRUE} by default).}

\item{retx}{bool, optional; \cr
logical value indicating whether the rotated variables / scores
should be returned (\eqn{TRUE} by default).}

\item{p}{integer, optional; \cr
oversampling parameter for \eqn{rsvd} (default \eqn{p=10}), see \code{\link{rsvd}}.}

\item{q}{integer, optional; \cr
number of additional power iterations for \eqn{rsvd} (default \eqn{q=1}), see \code{\link{rsvd}}.}

\item{rand}{bool, optional; \cr
if (\eqn{TRUE}), the \eqn{rsvd} routine is used, otherwise \eqn{svd} is used.}
}
\value{
\code{rpca} returns a list with class \eqn{rpca} containing the following components:
   \item{rotation}{  array_like; \cr
                     the rotation (eigenvectors); \eqn{(n, k)} dimensional array.
   }

   \item{eigvals}{  array_like; \cr
                    eigenvalues; \eqn{k} dimensional vector.
   }
   \item{sdev}{     array_like; \cr
                    standard deviations of the principal components; \eqn{k} dimensional vector.
   }
   \item{x}{        array_like; \cr
                    the scores / rotated data; \eqn{(m, k)} dimensional array.
   }
   \item{center, scale}{  array_like; \cr
                    the centering and scaling used.
   }
}
\description{
Fast computation of the principal components analysis using the randomized singular value decomposition.
}
\details{
Principal component analysis is an important linear dimension reduction technique.

Randomized PCA is computed via the randomized SVD algorithm (\code{\link{rsvd}}).
The computational gain is substantial, if the desired number of principal components
is relatively small, i.e. \eqn{k << min(m,n)}.

The print and summary method can be used to present the results in a nice format.
A scree plot can be produced with \code{\link{ggscreeplot}}. 
The individuals factor map can be produced with \code{\link{ggindplot}},
and a correlation plot with \code{\link{ggcorplot}}.

The predict function can be used to compute the scores of new observations. The data
will automatically be centered (and scaled if requested). This is not fully supported for
complex input matrices.
}
\note{
The principal components are not unique and only defined up to sign
(a constant of modulus one in the complex case) and so may differ between different
 PCA implementations.

Similar to \code{\link{prcomp}} the variances are computed with the usual divisor N - 1.


\itemize{
 \item [1] N. B. Erichson, S. Voronin, S. L. Brunton and J. N. Kutz. 2019.
       Randomized Matrix Decompositions Using {R}. 
       Journal of Statistical Software, 89(11), 1-48.
       \url{http://doi.org/10.18637/jss.v089.i11}.

  \item  [2] N. Halko, P. Martinsson, and J. Tropp.
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009).
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
}
}
\examples{

library('rsvd')
#
# Load Edgar Anderson's Iris Data
#
data('iris')

#
# log transform
#
log.iris <- log( iris[ , 1:4] )
iris.species <- iris[ , 5]

#
# Perform rPCA and compute only the first two PCs
#
iris.rpca <- rpca(log.iris, k=2)
summary(iris.rpca) # Summary
print(iris.rpca) # Prints the rotations

#
# Use rPCA to compute all PCs, similar to \\code{\\link{prcomp}}
#
iris.rpca <- rpca(log.iris)
summary(iris.rpca) # Summary
print(iris.rpca) # Prints the rotations
plot(iris.rpca) # Produce screeplot, variable and individuls factor maps.

}
\seealso{
\code{\link{ggscreeplot}}, \code{\link{ggindplot}},
         \code{\link{ggcorplot}}, \code{\link{plot.rpca}},
         \code{\link{predict}},   \code{\link{rsvd}}
}
\author{
N. Benjamin Erichson, \email{erichson@berkeley.edu}
}
