% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write.jagsfile.R
\name{write.jagsfile}
\alias{write.jagsfile}
\alias{write.JAGSfile}
\title{Write a complete JAGS model to a text file}
\usage{
write.jagsfile(
  runjags.object,
  file,
  remove.tags = TRUE,
  write.data = TRUE,
  write.inits = TRUE
)

write.JAGSfile(
  runjags.object,
  file,
  remove.tags = TRUE,
  write.data = TRUE,
  write.inits = TRUE
)
}
\arguments{
\item{runjags.object}{a valid (but not necessarily updated) runjags object to be saved to file.  No default.}

\item{file}{a filename to which the model will be written.  Note that any files already existing in that location will be overwritten with no warning (see \code{\link{new_unique}} for a way to generate unique filenames).  No default.}

\item{remove.tags}{should the runjags tags #data#, #inits#, #monitors#, #modules#, #factories#, #residual#, #fitted# and #response# be removed from the original model code before writing it to file?  If left in, these may create conflicts with the tags automatically added to the new file.}

\item{write.data}{should the data also be written to file?  If FALSE, the model may not run from the file without specifying a new source of data.}

\item{write.inits}{should the data also be written to file?  If FALSE, the model may not run from the file without specifying new initial values.}
}
\value{
Returns the filename that the model was saved to (invisibly)
}
\description{
Writes the JAGS model, data, initial values and monitored variables etc to a file.  The model can then be run using a call to \code{link{run.jags}} with the filename as the model argument.
}
\examples{
# Set up a model:
# y = m x + c, assuming normal observation errors for y:

# Simulate the data
X <- 1:100
Y <- rnorm(length(X), 2*X + 10, 1)

# Model in the JAGS format
model <- "model { 
for(i in 1 : N){ 
	Y[i] ~ dnorm(true.y[i], precision);
	true.y[i] <- (m * X[i]) + c
} 
m ~ dunif(-1000,1000)
c ~ dunif(-1000,1000) 
precision ~ dexp(1)
}"

# Data and initial values in a named list format, 
# with explicit control over the random number
# generator used for each chain (optional): 
data <- list(X=X, Y=Y, N=length(X))
inits1 <- list(m=1, c=1, precision=1,
.RNG.name="base::Super-Duper", .RNG.seed=1)
inits2 <- list(m=0.1, c=10, precision=1,
.RNG.name="base::Wichmann-Hill", .RNG.seed=2)

\dontrun{
# Compile the model but don't update it (sample=0):
compiled <- run.jags(model=model, monitor=c("m", "c", "precision"), 
data=data, n.chains=2, inits=list(inits1,inits2), sample=0)

# Save the complete model to a file:
filepath <- write.jagsfile(compiled, file='model.txt')

# And run the model from the file:
results <- run.jags(filepath)

}

}
\references{
Lunn D, Jackson C, Best N, Thomas A, Spiegelhalter D (2012). The BUGS book: A practical introduction to Bayesian analysis. CRC press; and Matthew J. Denwood (2016). runjags: An R Package Providing Interface Utilities, Model Templates, Parallel Computing Methods and Additional Distributions for MCMC Models in JAGS. Journal of Statistical Software, 71(9), 1-25. doi:10.18637/jss.v071.i09
}
\seealso{
\code{\link{read.jagsfile}} and \code{\link{run.jags}} for the reverse operation
}
\keyword{methods}
