% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aaa_generics.R
\docType{data}
\name{lung.saemix}
\alias{lung.saemix}
\title{NCCTG Lung Cancer Data, in SAEM format}
\format{
This data frame contains the following columns:
\describe{
\item{id}{subject index in file}
\item{inst}{institution code}
\item{time}{observation time since the beginning of follow-up}
\item{status}{0=alive, 1=dead}
\item{cens}{0=observed, 1=censored}
\item{sex}{patient gender (0=male, 1=female)}
\item{age}{age in years}
\item{ph.ecog}{ECOG performance score as rated by the physician. 0=asymptomatic,
1= symptomatic but completely ambulatory, 2= in bed <50\% of the day, 3= in bed > 50\% of the day but not bedbound,
4 = bedbound}
\item{ph.karno}{Karnofsky performance score (bad=0-good=100) rated by physician (\%)}
\item{pat.karno}{Karnofsky performance score (bad=0-good=100) rated by patient (\%)}
\item{meal.cal}{calories consumed at meals (cal)}
\item{wt.loss}{weight loss in last six months (pounds)}
}
}
\source{
Terry Therneau from the survival package in R
}
\usage{
lung.saemix
}
\description{
The \code{lung.saemix} contains survival data in patients with advanced lung cancer from the North Central Cancer Treatment Group.
Performance scores rate how well the patient can perform usual daily activities. This data is available in the survival library for R
and has been reformatted here for use in saemix (see details).
}
\details{
The data in the \code{lung.saemix} was reformatted from the lung cancer dataset (see data(cancer, package="survival")).
Patients with missing age, sex, institution or physician assessments were removed from the dataset. Status was recoded as 1 for death
and 0 for a censored event, and a censoring column was added to denote whether the patient was dead or alive at the time of
the last observation. For saemix, a line at time=0 was added for all subjects. Finally, subjects were numbered consecutively from 0 to 1.
}
\examples{
data(lung.saemix)

saemix.data<-saemixData(name.data=lung.saemix,header=TRUE,name.group=c("id"),
name.predictors=c("time","status","cens"),name.response=c("status"),
name.covariates=c("age", "sex", "ph.ecog", "ph.karno", "pat.karno", "wt.loss","meal.cal"),
units=list(x="days",y="",covariates=c("yr","","-","\%","\%","cal","pounds")))
weibulltte.model<-function(psi,id,xidep) {
  T<-xidep[,1]
  y<-xidep[,2] # events (1=event, 0=no event)
  cens<-which(xidep[,3]==1) # censoring times (subject specific)
  init <- which(T==0)
  lambda <- psi[id,1] # Parameters of the Weibull model
  beta <- psi[id,2]
  Nj <- length(T)
  ind <- setdiff(1:Nj, append(init,cens)) # indices of events
  hazard <- (beta/lambda)*(T/lambda)^(beta-1) # H'
  H <- (T/lambda)^beta # H
  logpdf <- rep(0,Nj) # ln(l(T=0))=0
  logpdf[cens] <- -H[cens] + H[cens-1] # ln(l(T=censoring time))
  logpdf[ind] <- -H[ind] + H[ind-1] + log(hazard[ind]) # ln(l(T=event time))
  return(logpdf)
}
saemix.model<-saemixModel(model=weibulltte.model,description="time model",modeltype="likelihood",
                psi0=matrix(c(1,2),ncol=2,byrow=TRUE,dimnames=list(NULL,  c("lambda","beta"))),
                transform.par=c(1,1),covariance.model=matrix(c(1,0,0,0),ncol=2, byrow=TRUE))
saemix.options<-list(seed=632545,save=FALSE,save.graphs=FALSE, displayProgress=FALSE)
\donttest{
tte.fit<-saemix(saemix.model,saemix.data,saemix.options)
}
# The fit from saemix using the above Weibull model may be compared 
# to the non-parametric KM estimate
\dontrun{
library(survival)
  lung.surv<-lung.saemix[lung.saemix$time>0,]
  lung.surv$status<-lung.surv$status+1
  Surv(lung.surv$time, lung.surv$status) # 1=censored, 2=dead
  f1 <- survfit(Surv(time, status) ~ 1, data = lung.surv)
  xtim<-seq(0,max(lung.saemix$time), length.out=200)
  estpar<-tte.fit@results@fixed.effects
  ypred<-exp(-(xtim/estpar[1])^(estpar[2]))
  plot(f1, xlab = "Days", ylab = "Overall survival probability")
  lines(xtim,ypred, col="red",lwd=2)
}
}
\references{
CL Loprinzi, JA Laurie, HS Wieand, JE Krook, PJ Novotny, JW Kugler, et al. (1994).
Prospective evaluation of prognostic variables from patient-completed questionnaires.
North Central Cancer Treatment Group. Journal of Clinical Oncology. 12(3):601-7.
}
\keyword{datasets}
