#' Calculate confidence interval for the mean
#'
#' @description
#' Calculate the confidence interval for the mean of each reporting unit.
#'
#' @param object A \code{sandwich.model} object generated by \code{\link{sandwich.model}}.
#' @param level The confidence level required. By default, \code{level} = .95.
#'
#' @return A \code{sandwich.ci} object that contains the lower and upper bounds of confidence intervals.

#' @usage sandwich.ci(object,
#'        level=.95)
#'
#' @import sf
#' @importFrom stats qt
#' @export
#'
#' @examples
#' data(hs.data)
#' hs.sw <- sandwich.model(object=hs.data, sampling.attr="Population", type="shp")
#' hs.sw.ci <- sandwich.ci(object=hs.sw, level=.95)
#'
#' @seealso \code{\link{autoplot}}, \code{\link{sandwich.model}}
#'
#' @name sandwich.ci
#'
# ---- End of roxygen documentation ----

sandwich.ci <- function(object, level=.95){
  #--------------------------- Check inputs ----------------------------------
  if (!is.element("mean", names(object$object)) |
      !is.element("se", names(object$object)) |
      !is.element("df", names(object$object))){
    stop("Should run the sandwich function first.")
  }

  #---------------- Calculate confidence intervals ----------------------
  t = qt(1-(1-level)/2, object$object$df)
  t[is.nan(t)] = 0
  object$object$ci.low = object$object$mean - t * object$object$se
  object$object$ci.up = object$object$mean + t * object$object$se

  output = list(object=object)
  class(output) <- "sandwich.ci"
  return(output)
}
