% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semEff-fun.R
\name{predEff}
\alias{predEff}
\title{Predict Effects}
\usage{
predEff(
  mod,
  newdata = NULL,
  effects = NULL,
  eff.boot = NULL,
  re.form = NA,
  type = c("link", "response"),
  interaction = NULL,
  use.raw = FALSE,
  ci.conf = 0.95,
  ci.type = "bca",
  digits = 3,
  bci.arg = NULL,
  parallel = "no",
  ncpus = NULL,
  cl = NULL,
  ...
)
}
\arguments{
\item{mod}{A fitted model object, or a list or nested list of such objects.}

\item{newdata}{An optional data frame of new values to predict, which should
contain all the variables named in \code{effects} or all those used to fit
\code{mod}.}

\item{effects}{A numeric vector of effects to predict, or a list or nested
list of such vectors. These will typically have been calculated using
\code{\link[=semEff]{semEff()}}, \code{\link[=bootEff]{bootEff()}}, or \code{\link[=stdEff]{stdEff()}}. Alternatively, a boot object
produced by \code{\link[=bootEff]{bootEff()}} can be supplied.}

\item{eff.boot}{A matrix of bootstrapped effects used to calculate confidence
intervals for predictions, or a list or nested list of such matrices. These
will have been calculated using \code{\link[=semEff]{semEff()}} or \code{\link[=bootEff]{bootEff()}}.}

\item{re.form}{For mixed models of class \code{"merMod"}, the formula for random
effects to condition on when predicting effects. Defaults to \code{NA}, meaning
random effects are averaged over. See \code{\link[=predict.merMod]{predict.merMod()}} for further
specification details.}

\item{type}{The type of prediction to return (for GLMs). Can be either
\code{"link"} (default) or \code{"response"}.}

\item{interaction}{An optional name of an interactive effect, for which to
return standardised effects for a 'main' continuous variable across
different values or levels of interacting variables (see Details).}

\item{use.raw}{Logical, whether to use raw (unstandardised) effects for all
calculations (if present).}

\item{ci.conf}{A numeric value specifying the confidence level for confidence
intervals on predictions (and any interactive effects).}

\item{ci.type}{The type of confidence interval to return (defaults to \code{"bca"}
– see Details). See \code{\link[=boot.ci]{boot.ci()}} for further specification details.}

\item{digits}{The number of significant digits to return for interactive
effects.}

\item{bci.arg}{A named list of any additional arguments to \code{\link[=boot.ci]{boot.ci()}},
excepting argument \code{index}.}

\item{parallel}{The type of parallel processing to use for calculating
confidence intervals on predictions. Can be one of \code{"snow"}, \code{"multicore"},
or \code{"no"} (for none – the default).}

\item{ncpus}{Number of system cores to use for parallel processing. If \code{NULL}
(default), all available cores are used.}

\item{cl}{Optional cluster to use if \code{parallel = "snow"}. If \code{NULL}
(default), a local cluster is created using the specified number of cores.}

\item{...}{Arguments to \code{\link[=stdEff]{stdEff()}}.}
}
\value{
A numeric vector of the predictions, or, if bootstrapped effects are
supplied, a list containing the predictions and the upper and lower
confidence intervals. Optional interactive effects may also be appended.
Predictions may also be returned in a list or nested list, depending on the
structure of \code{mod} (and other arguments).
}
\description{
Generate predicted values for SEM direct, indirect, or total
effects.
}
\details{
Generate predicted values for SEM direct, indirect, or total effects
on a response variable, which should be supplied to \code{effects}. These are
used in place of model coefficients in the standard prediction formula,
with values for predictors drawn either from the data used to fit the
original model(s) (\code{mod}) or from \code{newdata}. It is assumed that effects are
fully standardised; however, if this is not the case, then the same
centring and scaling options originally specified to \code{\link[=stdEff]{stdEff()}} should be
re-specified – which will then be used to standardise the data. If no
effects are supplied, standardised (direct) effects will be calculated from
the model and used to generate predictions. These predictions will equal
the model(s) fitted values if \code{newdata = NULL}, \code{unique.eff = FALSE}, and
\code{re.form = NULL} (where applicable).

Model-averaged predictions can be generated if averaged \code{effects} are
supplied to the model in \code{mod}, or, alternatively, if \code{weights} are
specified (passed to \code{\link[=stdEff]{stdEff()}}) and \code{mod} is a list of candidate models
(\code{effects} can also be passed using this latter method). For mixed model
predictions where random effects are included (e.g. \code{re.form = NULL}), the
latter approach should be used, otherwise the contribution of random
effects will be taken from the single model instead of (correctly) being
averaged over a candidate set.

If bootstrapped effects are supplied to \code{eff.boot} (or to \code{effects}, as
part of a boot object), bootstrapped predictions are calculated by
predicting from each effect. Confidence intervals can then be returned via
\code{\link[=bootCI]{bootCI()}}, for which the \code{type} should be appropriate for the original
form of bootstrap sampling (defaults to \code{"bca"}). If the number of
observations to predict is very large, parallel processing (via
\code{\link[=pSapply]{pSapply()}}) may speed up the calculation of intervals.

Predictions are always returned in the original (typically unstandardised)
units of the (link-transformed) response variable. For GLMs, they can be
returned in the response scale if \code{type = "response"}.

Additionally, if the name of an interactive effect is supplied to
\code{interaction}, standardised effects (and confidence intervals) can be
returned for effects of a continuous 'main' variable across different
values or levels of interacting variable(s). The name should be of the form
\code{"x1:x2..."}, containing all the variables involved and matching the name
of an interactive effect in the model(s) terms or in \code{effects}. The values
for all variables should be supplied in \code{newdata}, with the main continuous
variable being automatically identified as that having the most unique
values.
}
\examples{
# Predict effects (direct, total)
m <- shipley.sem
e <- shipley.sem.eff
dir <- getDirEff(e)
tot <- getTotEff(e)
f.dir <- predEff(m, effects = dir, type = "response")
f.tot <- predEff(m, effects = tot, type = "response")
f.dir$Live[1:10]
f.tot$Live[1:10]

# Using new data for predictors
d <- na.omit(shipley)
xn <- c("lat", "DD", "Date", "Growth")
seq100 <- function(x) seq(min(x), max(x), length = 100)
nd <- data.frame(sapply(d[xn], seq100))
f.dir <- predEff(m, nd, dir, type = "response")
f.tot <- predEff(m, nd, tot, type = "response")
f.dir$Live[1:10]
f.tot$Live[1:10]
# Add CIs
# dir.b <- getDirEff(e, "boot")
# tot.b <- getTotEff(e, "boot")
# f.dir <- predEff(m, nd, dir, dir.b, type = "response")
# f.tot <- predEff(m, nd, tot, tot.b, type = "response")

# Predict an interactive effect (e.g. Live ~ Growth * DD)
xn <- c("Growth", "DD")
d[xn] <- scale(d[xn])  # scale predictors (improves fit)
m <- lme4::glmer(Live ~ Growth * DD + (1 | site) + (1 | tree),
                 family = binomial, data = d)
nd <- with(d, expand.grid(
  Growth = seq100(Growth),
  DD = mean(DD) + c(-sd(DD), sd(DD))  # two levels for DD
))
f <- predEff(m, nd, type = "response", interaction = "Growth:DD")
f$fit[1:10]
f$interaction
# Add CIs (need to bootstrap model...)
# system.time(B <- bootEff(m, R = 1000, ran.eff = "site"))
# f <- predEff(m, nd, B, type = "response", interaction = "Growth:DD")

# Model-averaged predictions (several approaches)
m <- shipley.growth  # candidate models (list)
w <- runif(length(m), 0, 1)  # weights
e <- stdEff(m, w)  # averaged effects
f1 <- predEff(m[[1]], effects = e)  # pass avg. effects
f2 <- predEff(m, weights = w)  # pass weights argument
f3 <- avgEst(predEff(m), w)  # use avgEst function
stopifnot(all.equal(f1, f2))
stopifnot(all.equal(f2, f3))

# Compare model fitted values: predEff() vs. fitted()
m <- shipley.sem$Live
f1 <- predEff(m, unique.eff = FALSE, re.form = NULL, type = "response")
f2 <- fitted(m)
stopifnot(all.equal(f1, f2))

# Compare predictions using standardised vs. raw effects (same)
f1 <- predEff(m)
f2 <- predEff(m, use.raw = TRUE)
stopifnot(all.equal(f1, f2))
}
\seealso{
\code{\link[=predict]{predict()}}
}
