#' @title Set the Colors of Selected Edges
#'
#' @description Set the colors of
#' selected edges.
#'
#' @details Modified a [qgraph::qgraph]
#' object generated by
#' [semPlot::semPaths] and change the
#' colors of selected edges.
#'
#' ## Setting the value of `color_list`
#'
#' This argument can be set in two ways.
#'
#' For a named vector, the name of an
#' element should be the path as
#' specified by [lavaan::model.syntax]
#' or as appeared in
#' [lavaan::parameterEstimates()].
#'
#' For example, to change the color of the
#' path regressing `y` on `x`, the name
#' should be `"y ~ x"`. To change the
#' color of the covariance between `x1`
#' and `x2`, the name should be `"x1 ~~
#' x2"`. Therefore, `c("y ~ x1" = "red",
#' "x1 ~~ x2" = "blue")` changes the
#' colors of the path from `x1` to `y`
#' and the covariance between `x1` and
#' `x2` to `"red"` and `"blue"`,
#' respectively.
#'
#' The order of the two nodes *may*
#' matter for covariances. Therefore, if
#' the attribute of a covariance is not
#' changed, try switching the order of
#' the two nodes.
#'
#' For a list of named lists, each named
#' list should have three named values:
#' `from`, `to`, and `new_color`. The
#' attribute of the edge from `from` to
#' `to` will be set to `new_color`.
#'
#' The second approach is no longer
#' recommended, though kept for backward
#' compatibility.
#'
#' @return A [qgraph::qgraph] based on
#' the original one, with colors for
#' selected edges changed.
#'
#' @param semPaths_plot A
#' [qgraph::qgraph] object generated by
#' [semPlot::semPaths], or a similar
#' qgraph object modified by other
#' [semptools] functions.
#'
#' @param color_list A named vector or a
#' list of named list. See the Details
#' section on how to set this argument.
#'
#' @examples
#' mod_pa <-
#'   'x1 ~~ x2
#'    x3 ~  x1 + x2
#'    x4 ~  x1 + x3
#'   '
#' fit_pa <- lavaan::sem(mod_pa, pa_example)
#' lavaan::parameterEstimates(fit_pa)[, c("lhs", "op", "rhs", "est", "pvalue")]
#' m <- matrix(c("x1",   NA,   NA,
#'                 NA, "x3", "x4",
#'               "x2",   NA,   NA), byrow = TRUE, 3, 3)
#' p_pa <- semPlot::semPaths(fit_pa, whatLabels="est",
#'             style = "ram",
#'             nCharNodes = 0, nCharEdges = 0,
#'             layout = m)
#'
#' my_color_vector <- c("x2 ~~ x1" = "red",
#'                      "x4 ~ x1" = "blue")
#'
#' p_pa2v <- set_edge_color(p_pa, my_color_vector)
#' plot(p_pa2v)
#'
#' my_color_list <- list(list(from = "x1", to = "x2", new_color = "red"),
#'                      list(from = "x1", to = "x4", new_color =  "blue"))
#'
#' p_pa2l <- set_edge_color(p_pa, my_color_list)
#' plot(p_pa2l)
#'
#' @export

set_edge_color <- function(semPaths_plot,
                      color_list = NULL) {

    if (is.null(color_list)) {
        stop("color_list not specified.")
      }
    if (missing(semPaths_plot)) {
        stop("semPaths_plot not specified.")
      } else {
        if (!inherits(semPaths_plot, "qgraph")) {
            stop("semPaths_plot is not a qgraph object.")
          }
      }

    # Fix the names
    color_list_fixed <- to_new_value(color_list,
                                     original_name = "new_color")

    out <- set_edge_attribute(semPaths_plot = semPaths_plot,
                              values = color_list_fixed,
                              attribute_name = "color")

    out
  }
