% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_clade_phyglm.R
\name{tree_clade_phyglm}
\alias{tree_clade_phyglm}
\title{Interaction between phylogenetic uncertainty and influential clade detection - Phylogenetic Logistic Regression}
\usage{
tree_clade_phyglm(formula, data, phy, clade.col, n.species = 5, n.sim = 100,
  n.tree = 2, btol = 50, track = TRUE, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'multiPhylo', see ?\code{ape}).}

\item{clade.col}{The column in the provided data frame which specifies the
clades (a character vector with clade names).}

\item{n.species}{Minimum number of species in a clade for the clade to be
included in the leave-one-out deletion analysis. Default is \code{5}.}

\item{n.sim}{Number of simulations for the randomization test.}

\item{n.tree}{Number of times to repeat the analysis with n different trees picked 
randomly in the multiPhylo file.
If NULL, \code{n.tree} = 2}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phyloglm}}
}
\value{
The function \code{clade_phyglm} returns a list with the following
components:

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{lambda}) for the full model
without deleted species.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade for a tree iteration. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DIFintercept}), the percentage of change
in intercept compared to the full model (\code{intercept.perc}) and intercept
p-value (\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter 
(e.g. \code{kappa} or \code{lambda}, depending on the phylogenetic model used) 
are reported.

\code{null.dist}: A data frame with estimates for the null distributions
for all clades analysed.

\code{data}: Original full dataset.

\code{errors}: Clades and/or trees where deletion resulted in errors.
}
\description{
Estimate the impact on model estimates of phylogenetic logistic regression after 
removing clades from the analysis and evaluating uncertainty in trees topology.
}
\details{
Currently only logistic regression using the "logistic_MPLE"-method from
\code{phyloglm} is implemented.

This function sequentially removes one clade at a time, fits a phylogenetic
logistic regression model using \code{\link[phylolm]{phyloglm}} and stores the
results. The impact of of a specific clade on model estimates is calculated by the
comparison between the full model (with all species) and the model without 
the species belonging to a clade. It repeats this operation using n trees, 
randomly picked in a multiPhylo file.

Additionally, to account for the influence of the number of species on each 
clade (clade sample size), this function also estimates a null distribution of slopes
expected for the number of species in a given clade. This is done by fitting
models without the same number of species in the given clade. 
 The number of simulations to be performed is set by 'n.sim'. To test if the 
 clade influence differs from the null expectation for a clade of that size, 
 a randomization test can be performed using 'summary(x)'. 


\code{clade_phyglm} detects influential clades based on
difference in intercept and/or slope when removing a given clade compared
to the full model including all species. This is done for n trees in the multiphylo file.

Currently, this function can only implements simple logistic models (i.e. \eqn{trait~
predictor}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
# Simulate Data:
set.seed(6987)
mphy = rmtree(150, N = 30)
x = rTrait(n=1,phy=mphy[[1]])
X = cbind(rep(1,150),x)
y = rbinTrait(n=1,phy=mphy[[1]], beta=c(-1,0.5), alpha=.7 ,X=X)
cla <- rep(c("A","B","C","D","E"), each = 30)
dat = data.frame(y, x, cla)
# Run sensitivity analysis:
tree_clade <- tree_clade_phyglm(y ~ x, phy = mphy, data = dat, 
n.tree = 10, n.sim = 10, clade.col = "cla")
# To check summary results and most influential clades:
summary(tree_clade)
# Visual diagnostics for clade removal:
sensi_plot(tree_clade)
# Specify which clade removal to plot:
sensi_plot(tree_clade, "B")
sensi_plot(tree_clade, "C", graphs = 2)
sensi_plot(tree_clade, "D", graphs = 2) 
}
\dontshow{
set.seed(6987)
mphy = rmtree(150, N = 30)
x = rTrait(n=1,phy=mphy[[1]])
X = cbind(rep(1,150),x)
y = rbinTrait(n=1,phy=mphy[[1]], beta=c(-1,0.5), alpha=.7 ,X=X)
cla <- rep(c("A","B","C","D","E"), each = 30)
dat = data.frame(y, x, cla)
# Run sensitivity analysis:
tree_clade <- tree_clade_phyglm(y ~ x, phy = mphy, data = dat, 
                               n.tree = 2, n.sim = 1, clade.col = "cla")
# To check summary results and most influential clades:
summary(tree_clade)
# Visual diagnostics for clade removal:
sensi_plot(tree_clade)
}
}
\references{
Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phyloglm}}, \code{\link[sensiPhy]{tree_phyglm}},
 \code{\link{clade_phyglm}}, \code{\link{tree_clade_phylm}},
\code{\link{sensi_plot}}
}
\author{
Gustavo Paterno, Caterina Penone & Gijsbert D.A. Werner
}
