
#' @title Execute an expression safely in server
#'
#' @description Execute an expression without generating an error,
#'  instead display the error to the user in an alert.
#'
#' @param expr Expression to evaluate
#' @param title Title to display in the alert in case of error.
#' @param message Message to display below title.
#' @param include_error Include the error message generated by R.
#' @param error_return Value to return in case of error.
#' @param session Shiny session.
#'
#' @return Result of \code{expr} if no error, otherwise the value of
#'  \code{error_return} (\code{NULL} by default to use \code{\link[shiny]{req}
#'  in other reactive context}).
#' @export
#'
#' @importFrom htmltools tags
#'
#' @example examples/execute_safely.R
execute_safely <- function(expr, title = "Error",
                           message = "An error occured, detail below:",
                           include_error = TRUE,
                           error_return = NULL,
                           session = shiny::getDefaultReactiveDomain()) {
  tryCatch(
    expr = expr,
    error = function(e) {
      if (isTRUE(include_error)) {
        message <- tags$div(
          style= "text-align: left;",
          message,
          tags$br(),
          tags$br(),
          tags$pre(
            style = "white-space:normal;",
            tags$code(
              as.character(e$message)
            )
          )
        )
      }
      message("Error: ", e$message)
      sendSweetAlert(
        session = session,
        title = title,
        text = message,
        html = TRUE,
        type = "error"
      )
      return(error_return)
    }
  )
}

