% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shrinkTVPVAR.R
\name{shrinkTVPVAR}
\alias{shrinkTVPVAR}
\title{Markov Chain Monte Carlo (MCMC) for TVP-VAR-SV models with shrinkage}
\usage{
shrinkTVPVAR(
  y,
  p = 1,
  mod_type = "double",
  const = TRUE,
  niter = 5000,
  nburn = round(niter/2),
  nthin = 1,
  display_progress = TRUE,
  TVP_params_beta = list(),
  TVP_params_sigma = list()
)
}
\arguments{
\item{y}{matrix or data frame containing the time series data. The rows correspond to the time
points and the columns to the variables.}

\item{p}{positive integer indicating the number of lags in the VAR model. The default value is 1.}

\item{mod_type}{character string that reads either \code{"triple"}, \code{"double"} or \code{"ridge"}.
Determines whether the triple gamma, double gamma or ridge prior are used for \code{theta_sr} and \code{beta_mean}.
The default is "double".}

\item{const}{logical value indicating whether a constant should be included in the model. The default value is \code{TRUE}.}

\item{niter}{positive integer, indicating the number of MCMC iterations
to perform, including the burn-in. Has to be larger than or equal to \code{nburn} + 2. The default value is 10000.}

\item{nburn}{non-negative integer, indicating the number of iterations discarded
as burn-in. Has to be smaller than or equal to \code{niter} - 2. The default value is \code{round(niter / 2)}.}

\item{nthin}{positive integer, indicating the degree of thinning to be performed. Every \code{nthin} draw is kept and returned.
The default value is 1, implying that every draw is kept.}

\item{display_progress}{logical value indicating whether the progress bar and other informative output should be
displayed. The default value is \code{TRUE}.}

\item{TVP_params_beta}{\emph{optional} named list containing hyperparameter values for the TVP prior of the beta_mean matrix.
Not all have to be supplied, with those missing being replaced by the default values. Any list elements that are misnamed will be ignored and a warning will be thrown.
Can either be a list of depth 1, which results in all equations having the same hyperparameters, or a list of lists of length \code{m}, where each sub-list contains the
hyperparameters for the respective equation. If sub-lists are provided, they can be unnamed, in which case the order of the elements is assumed to be the same as the order of the equations in the data.
Alternatively, they can be named \code{eq1}, \code{eq2}, ..., \code{eqm}.
The meaning of the hyperparameters is the same as in \code{\link[shrinkTVP]{shrinkTVP}}, as this function is used to sample the TVP coefficients.
A key difference lies in how the adaptive MH arguments are provided. In this function, the adaptive MH arguments are provided as vectors of length 4,
where the elements control the MH steps in following order: \code{a_xi}, \code{a_tau}, \code{c_xi}, \code{c_tau}.
E.g. if \code{adaptive = c(FALSE, TRUE, TRUE, FALSE)}, then the MH step for \code{a_xi} and \code{c_tau} are non-adaptive, while the MH step for \code{a_tau} and \code{c_xi} are adaptive.
The following elements can be supplied:
\itemize{
\item \code{e1}: positive, real number. The default value is 0.5.
\item \code{e2}: positive, real number. The default value is 0.001.
\item \code{d1}: positive, real number. The default value is 0.5.
\item \code{d2}: positive, real number. The default value is 0.001.
\item \code{beta_a_xi}: positive, real number. The default value is 10.
\item \code{beta_a_tau}: positive, real number. The default value is 10.
\item \code{alpha_a_xi}: positive, real number. The default value is 5.
\item \code{alpha_a_tau}: positive, real number. The default value is 5.
\item \code{beta_c_xi}: positive, real number. The default value is 2.
\item \code{beta_c_tau}: positive, real number. The default value is 2.
\item \code{alpha_c_xi}: positive, real number. The default value is 5.
\item \code{alpha_c_tau}: positive, real number. The default value is 5.
\item \code{a_tuning_par_xi}: positive, real number. The default value is 1.
\item \code{a_tuning_par_tau}: positive, real number. The default value is 1.
\item \code{c_tuning_par_xi}: positive, real number. The default value is 1.
\item \code{c_tuning_par_tau}: positive, real number. The default value is 1.
\item \code{learn_a_xi}: logical. The default value is \code{TRUE}.
\item \code{learn_a_tau}: logical. The default value is \code{TRUE}.
\item \code{a_eq_c_xi}: logical. The default value is \code{FALSE}.
\item \code{a_eq_c_tau}: logical. The default value is \code{FALSE}.
\item \code{a_xi}: positive, real number. The default value is 0.1.
\item \code{a_tau}: positive, real number. The default value is 0.1.
\item \code{learn_c_xi}: logical. The default value is \code{TRUE}.
\item \code{learn_c_tau}: logical. The default value is \code{TRUE}.
\item \code{c_xi}: positive, real number. The default value is 0.1.
\item \code{c_tau}: positive, real number. The default value is 0.1.
\item \code{learn_kappa2_B}: logical. The default value is \code{TRUE}.
\item \code{learn_lambda2_B}: logical. The default value is \code{TRUE}.
\item \code{kappa2_B}: positive, real number. The default value is 20.
\item \code{lambda2_B}: positive, real number. The default value is 20.
\item \code{Bsigma_sv}: positive, real number. The default value is 1.
\item \code{a0_sv}: positive, real number. The default value is 5.
\item \code{b0_sv}: positive, real number. The default value is 1.5.
\item \code{bmu}: real number. The default value is 0.
\item \code{Bmu}: positive, real number. The default value is 1.
\item \code{adaptive}: logical vector of length 4. The default value is \code{rep(TRUE, 4)}.
\item \code{target_rates}: numeric vector of length 4. The default value is \code{rep(0.44, 4)}.
\item \code{batch_sizes}: numeric vector of length 4. The default value is \code{rep(50, 4)}.
\item \code{max_adapts}: numeric vector of length 4. The default value is \code{rep(0.01, 4)}.
}}

\item{TVP_params_sigma}{\emph{optional} named list containing hyperparameter values for the TVP prior of the Sigma matrix.
The structure is the same as for \code{TVP_params_beta}. The default values are the same as for \code{TVP_params_beta}.}
}
\value{
The value returned is a \code{shrinkTVPVAR} object containing:
\item{\code{beta}}{an object of class \code{"mcmc.tvp.var"} containing the MCMC draws of the VAR coefficients.}
\item{\code{beta_mean}}{an object of class \code{"mcmc.var"} containing the MCMC draws of beta_mean for the VAR coefficients.}
\item{\code{theta_sr}}{an object of class \code{"mcmc.var"} containing the MCMC draws of theta_sr for the VAR coefficients.}
\item{\code{xi2}}{an object of class \code{"mcmc.var"} containing the MCMC draws of xi2 for the VAR coefficients.}
\item{\code{c_xi}}{an object of class \code{"mcmc"} containing the MCMC draws of c_xi for the VAR coefficients.}
\item{\code{kappa2}}{an object of class \code{"mcmc.var"} containing the MCMC draws of kappa2 for the VAR coefficients.}
\item{\code{kappa2_B}}{an object of class \code{"mcmc"} containing the MCMC draws of kappa2_B for the VAR coefficients.}
\item{\code{a_xi}}{an object of class \code{"mcmc"} containing the MCMC draws of a_xi for the VAR coefficients.}
\item{\code{tau2}}{an object of class \code{"mcmc.var"} containing the MCMC draws of tau2 for the VAR coefficients.}
\item{\code{c_tau}}{an object of class \code{"mcmc"} containing the MCMC draws of c_tau for the VAR coefficients.}
\item{\code{lambda2}}{an object of class \code{"mcmc.var"} containing the MCMC draws of lambda2 for the VAR coefficients.}
\item{\code{lambda2_B}}{an object of class \code{"mcmc"} containing the MCMC draws of lambda2_B for the VAR coefficients.}
\item{\code{a_tau}}{an object of class \code{"mcmc"} containing the MCMC draws of a_tau for the VAR coefficients.}
\item{\code{Sigma}}{an object of class \code{"mcmc.tvp.var"} containing the MCMC draws of the covariance matrices.}
\item{\code{pred_objs}}{a list containing objects required for prediction methods to work.}
\item{\code{beta_consts}}{a list of \code{mcmc.tvp} objects containing the MCMC draws of the intercepts.}
\item{\code{data}}{a list containing the input data, as well as the synthetic "covariates" used for estimation.}
Note that only the values pertaining to the VAR coefficients are returned.
The values for the variance-covariance matrix are not returned.

To display the output, use \code{plot} on various elements of the list, as well as the \code{\link{TV_heatmap}},
\code{\link{density_plotter}} and \code{\link{state_plotter}} function. Many functions that can be applied to \code{coda::mcmc} objects
(e.g. \code{coda::acfplot}) can be applied to all output elements that are \code{coda} compatible.
}
\description{
\code{shrinkTVPVAR} samples from the joint posterior distribution of the parameters of a TVP-VAR-SV
model with shrinkage as described in  Cadonna et al. (2020) and returns the MCMC draws.
The model can be written as:
\deqn{Y_t = c_t + \Phi_{1,t} Y_{t-1} + \Phi_{2,t} Y_{t-2} + \cdots + \Phi_{p,t} Y_{t-p} + \epsilon_t}
where \eqn{\epsilon_t \sim \mathcal{N}_m(0, \Sigma_t)}.
}
\details{
The elements of the VAR coefficients \eqn{\Phi_{i,t}} are assumed to follow component-wise
random walk.

For further details concerning the algorithms and the model please refer to the paper by Cadonna et al. (2020).
}
\examples{
\donttest{
set.seed(123)
sim <- simTVPVAR(p = 2)
data <- sim$data

res <- shrinkTVPVAR(data, p = 2)

# Visualize the results
plot(res)

plot(res$theta_sr)

# Change prior to triple gamma
res2 <- shrinkTVPVAR(data, p = 2, mod_type = "triple")

# Modify the hyperparameter setup
# Estimate under hierarchical horseshoe prior
hyperparam <- list(learn_a_xi = FALSE, learn_c_xi = FALSE,
                  learn_a_tau = FALSE, learn_c_tau = FALSE,
                  a_xi = 0.5, c_xi = 0.5, a_tau = 0.5, c_tau = 0.5)

res3 <- shrinkTVPVAR(data, p = 2, mod_type = "triple",
                    TVP_params_beta = hyperparam,
                    TVP_params_sigma = hyperparam)

# Can also specify different hyperparameters for each equation
# gen_TVP_params() is a helper function and returns a
# list of lists (if for_each_eq = TRUE) where each sub-list
# contains the hyperparameters for the respective equation
hyperparam2 <- gen_TVP_params(m = 3, for_each_eq = TRUE)

hyperparam2[[1]]$learn_a_xi <- FALSE
hyperparam2[[1]]$a_xi <- 0.5

res4 <- shrinkTVPVAR(data, p = 2, mod_type = "triple",
                     TVP_params_beta = hyperparam2)

# Now, a_xi is only fixed in first equation
plot(res4$a_xi)
}

}
\references{
Cadonna, A., Frühwirth-Schnatter, S., & Knaus, P. (2020). "Triple the Gamma—A Unifying Shrinkage Prior for Variance and Variable Selection in Sparse State Space and TVP Models."
\emph{Econometrics}, 8(2), 20. <doi:10.3390/econometrics8020020>

Knaus, P., Bitto-Nemling, A., Cadonna, A., & Frühwirth-Schnatter, S. (2021) "Shrinkage in the Time-Varying Parameter Model Framework Using the \code{R} Package \code{shrinkTVP}."
\emph{Journal of Statistical Software} 100(13), 1–32. <doi:10.18637/jss.v100.i13>
}
\seealso{
\code{\link{TV_heatmap}} \code{\link{density_plotter}} \code{\link{state_plotter}}
}
\author{
Peter Knaus \email{peter.knaus@wu.ac.at}
}
