#' @title Sigmoidal fit function
#'
#' @param dataInput a data frame composed of two columns. One is for time and the other is for intensity. Should be normalized data generated by normalizeData.
#' @param tryCounter the number of times the data is fit using the fitting algorithm.
#' @param startList the initial set of parameters that algorithm tries to fit. The parameters are 'maximumValue' that represents the maximum value that the function can take,  'slopeValue' that represents the slope in normalized y axis, and 'midPointValue' that represents the midpoint.
#' @param lowerBounds the lower bounds for the randomly generated start parameters.
#' @param upperBounds the upper bounds for the randomly generated start parameters.
#' @param min_Factor the minimum step size in the iterations used by the fitting algorithm.
#' @param n_iterations the maximum number of iterations used by the fitting algorithm.
#'
#' @description Fits a sigmoidal curve to given data by using likelihood maximization (LM) and gives the parameters (maximum, slope and midpoint) describing the sigmoidal fit as output. It also provides information about the goodness of fit such as AIC, BIC, residual sum of squares, and log likelihood.
#' @return Returns fitted parameters for the sigmoidal model.
#' @export
#'
#' @examples
#'time=seq(3,24,0.5)
#'
#'#simulate intensity data and add noise
#'noise_parameter=0.1
#'intensity_noise=stats::runif(n = length(time),min = 0,max = 1)*noise_parameter
#'intensity=sigmoidalFitFormula(time, maximum=4, slope=1, midPoint=8)
#'intensity=intensity+intensity_noise
#'
#'dataInput=data.frame(intensity=intensity,time=time)
#'normalizedInput = normalizeData(dataInput)
#'parameterVector<-sigmoidalFitFunction(normalizedInput,tryCounter=2)
#'
#'#Check the results
#'if(parameterVector$isThisaFit){
#'intensityTheoretical=sigmoidalFitFormula(time,
#'                                         maximum=parameterVector$maximum_Estimate,
#'                                         slope=parameterVector$slope_Estimate,
#'                                         midPoint=parameterVector$midPoint_Estimate)
#'
#'comparisonData=cbind(dataInput,intensityTheoretical)
#'
#'require(ggplot2)
#'ggplot(comparisonData)+
#'  geom_point(aes(x=time, y=intensity))+
#'  geom_line(aes(x=time,y=intensityTheoretical))+
#'  expand_limits(x = 0, y = 0)}
#'
#'if(!parameterVector$isThisaFit){print(parameterVector)}
#'
sigmoidalFitFunction<-function(dataInput,
                               tryCounter,
                               startList=list(maximum = 1, slope = 36, midPoint = 0.3333333),
                               lowerBounds=c(maximum=0.3, slope=0.00001,  midPoint=0.3125-0.8333333),
                               upperBounds=c(maximum=1.5, slope=180,  midPoint=0.3125+0.8333333),
                               min_Factor=1/2^20,
                               n_iterations=1000)
{

  isalist=(is.list(dataInput) & !is.data.frame(dataInput))
  if(isalist){dataFrameInput=dataInput$timeIntensityData}
  isadataframe=(is.data.frame(dataInput))
  if(isadataframe){dataFrameInput=dataInput}

  if(tryCounter==1){counterDependentStartList=startList}
  if(tryCounter!=1){
    randomVector=stats::runif(length(startList), 0, 1)
    names(randomVector)<-c("maximum", "slope", "midPoint")
    counterDependentStartVector=randomVector*(upperBounds-lowerBounds)+lowerBounds
    counterDependentStartList=as.list(counterDependentStartVector)}

  theFitResult <- try(minpack.lm::nlsLM(intensity ~ sicegar::sigmoidalFitFormula(time, maximum, slope, midPoint),
                                        dataFrameInput,
                                        start=counterDependentStartList,
                                        control = list(maxiter = n_iterations, minFactor = min_Factor),
                                        lower = lowerBounds,
                                        upper = upperBounds,
                                        trace=F),silent = TRUE)

  if(class(theFitResult)!="try-error")
  {
    parameterMatrix=summary(theFitResult)$parameters
    colnames(parameterMatrix)<-c("Estimate","Std_Error","t_value","Pr_t")

    parameterVector=c(t(parameterMatrix))
    names(parameterVector)<- c("maximum_N_Estimate","maximum_Std_Error","maximum_t_value","maximum_Pr_t",
                               "slope_N_Estimate","slope_Std_Error","slope_t_value","slope_Pr_t",
                               "midPoint_N_Estimate","midPoint_Std_Error","midPoint_t_value","midPoint_Pr_t")

    parameterVector<-c(parameterVector,
                       residual_Sum_of_Squares=sum((as.vector(stats::resid(theFitResult)))^2)[1],
                       log_likelihood=as.vector(stats::logLik(theFitResult))[1],
                       AIC_value=as.vector(stats::AIC(theFitResult))[1],
                       BIC_value=as.vector(stats::BIC(theFitResult))[1])

    parameterList=as.list(parameterVector)
    parameterList$isThisaFit=TRUE
    parameterList$startVector=counterDependentStartList
    if(isalist){parameterList$dataScalingParameters=as.list(dataInput$dataScalingParameters)}
    parameterList$model="sigmoidal"

    parameterDf=as.data.frame(parameterList)
    #Renormalize Parameters
    parameterDf=sigmoidalRenormalizeParameters(parameterDf,isalist)

  }

  if(class(theFitResult)=="try-error")
  {
    parameterVector=rep(NA, 12)
    names(parameterVector)<- c("maximum_N_Estimate","maximum_Std_Error","maximum_t_value","maximum_Pr_t",
                               "slope_N_Estimate","slope_Std_Error","slope_t_value","slope_Pr_t",
                               "midPoint_N_Estimate","midPoint_Std_Error","midPoint_t_value","midPoint_Pr_t")

    parameterVector<-c(parameterVector,
                       residual_Sum_of_Squares=Inf,
                       log_likelihood=NA,
                       AIC_value=NA,
                       BIC_value=NA)

    parameterList=as.list(parameterVector)
    parameterList$isThisaFit=FALSE
    parameterList$startVector=counterDependentStartList
    if(isalist){parameterList$dataScalingParameters=as.list(dataInput$dataScalingParameters)}
    parameterList$model="sigmoidal"

    parameterDf=as.data.frame(parameterList)
    #Renormalize Parameters
    parameterDf=sigmoidalRenormalizeParameters(parameterDf,isalist)


  }

  return(parameterDf)
}

#**************************************
#' @title sigmoidalFitFormula
#'
#' @param x  the "time" (time) column of the dataframe.
#' @param maximum the maximum value that the sigmoidal function can reach.
#' @param slope  the slope of the sigmoidal function at the steppest point.
#' @param midPoint  the x axis value of the steppest point in the function.
#'
#' @description Calculates intesities for given time points (x) by using sigmoidal fit model and parameters (maximum, slope, and midpoint).
#' @return Returns the predicted intensities for given time points with the given sigmoidal fit parameters.
#'
#' @examples
#'
#'time=seq(3,24,0.5)
#'
#'#simulate intensity data and add noise
#'noise_parameter=0.1
#'intensity_noise=stats::runif(n = length(time),min = 0,max = 1)*noise_parameter
#'intensity=sigmoidalFitFormula(time, maximum=4, slope=1, midPoint=8)
#'intensity=intensity+intensity_noise
#'
#'dataInput=data.frame(intensity=intensity,time=time)
#'normalizedInput = normalizeData(dataInput)
#'parameterVector<-sigmoidalFitFunction(normalizedInput,tryCounter=2)
#'
#'#Check the results
#'if(parameterVector$isThisaFit){
#'intensityTheoretical=sigmoidalFitFormula(time,
#'                                         maximum=parameterVector$maximum_Estimate,
#'                                         slope=parameterVector$slope_Estimate,
#'                                         midPoint=parameterVector$midPoint_Estimate)
#'
#'comparisonData=cbind(dataInput,intensityTheoretical)
#'
#'require(ggplot2)
#'ggplot(comparisonData)+
#'  geom_point(aes(x=time, y=intensity))+
#'  geom_line(aes(x=time,y=intensityTheoretical))+
#'  expand_limits(x = 0, y = 0)}
#'
#'if(!parameterVector$isThisaFit){print(parameterVector)}
#'
#'
#'
#' @export
sigmoidalFitFormula<-function(x, maximum, slope, midPoint){
  y=(0 + (maximum - 0)/(1 + exp((-slope)*(x - midPoint))));
  return(y)}
#**************************************


#**************************************
# @title sigmoidalRenormalizeParameters (This is an internal function)
# @param parametersDf it is the parameter data frame generated by sigmoidal fit function
#        includes the parameters named
#        *maximum_N_Estimate (normalized Maximum Estimate)
#        *slope_N_Estimate (normalzied Slope Estimate)
#        *midPoint_N_Estimate (normalized Midpoint Estimate)
#        *dataScalingParameters.intensityRatio the range of initial unnormalized intensity. Provieded if the data is normalized
#        *parameterDF$dataScalingParameters.intensityMin the minimum of unnormalized intensity. Provieded if the data is normalized
#        *parameterDF$dataScalingParameters.timeRatio tha maximum time that the experiment reach. Provieded if the data is normalized
# @param isalist defines if the input is provided as a list (i.e normalized) or as a data frame (i.e not normalized)
# @details If the fit was done in normalized data frame then the found fit parameters will belong to normalized data.
#          This function generates unnormalized counterparts of those parameters.
sigmoidalRenormalizeParameters<-function(parameterDF,isalist)
{
  if(isalist){
    parameterDF$maximum_Estimate=parameterDF$maximum_N_Estimate*parameterDF$dataScalingParameters.intensityRatio+parameterDF$dataScalingParameters.intensityMin
    parameterDF$slope_Estimate=parameterDF$slope_N_Estimate/parameterDF$dataScalingParameters.timeRatio
    parameterDF$midPoint_Estimate=parameterDF$midPoint_N_Estimate*parameterDF$dataScalingParameters.timeRatio
  }
  if(!isalist){
    parameterDF$maximum_Estimate=parameterDF$maximum_N_Estimate
    parameterDF$slope1_Estimate=parameterDF$slope_N_Estimate
    parameterDF$midPoint_Estimate=parameterDF$midPoint_N_Estimate
  }
  return(parameterDF)
}
#**************************************
