% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-lin-growth.R
\name{SimSSMLinGrowth}
\alias{SimSSMLinGrowth}
\title{Simulate Data from the
Linear Growth Curve Model}
\usage{
SimSSMLinGrowth(
  n,
  time,
  mu0,
  sigma0_l,
  theta_l,
  type = 0,
  x = NULL,
  gamma = NULL,
  kappa = NULL
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{time}{Positive integer.
Number of time points.}

\item{mu0}{Numeric vector.
A vector of length two.
The first element is the mean of the intercept,
and the second element is the mean of the slope.}

\item{sigma0_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(sigma0))})
of the covariance matrix
of the intercept and the slope.}

\item{theta_l}{Numeric.
Square root of the common measurement error variance.}

\item{type}{Integer.
State space model type.
See Details for more information.}

\item{x}{List.
Each element of the list is a matrix of covariates
for each individual \code{i} in \code{n}.
The number of columns in each matrix
should be equal to \code{time}.}

\item{gamma}{Numeric matrix.
Matrix linking the covariates to the latent variables
at current time point
(\eqn{\boldsymbol{\Gamma}}).}

\item{kappa}{Numeric matrix.
Matrix linking the covariates to the observed variables
at current time point
(\eqn{\boldsymbol{\kappa}}).}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
Each element of \code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ID numbers with length \code{l},
where \code{l} is the value of the function argument \code{time}.
\item \code{time}: A vector time points of length \code{l}.
\item \code{y}: A \code{l} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{l} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{l} by \code{j} matrix of values for the covariates
(when covariates are included).
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data from the
linear growth curve model.
}
\details{
\subsection{Type 0}{

The measurement model is given by
\deqn{
    Y_{i, t}
    =
    \left(
    \begin{array}{cc}
      1 & 0 \\
    \end{array}
    \right)
    \left(
    \begin{array}{c}
      \eta_{0_{i, t}} \\
      \eta_{1_{i, t}} \\
    \end{array}
    \right)
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    0,
    \theta
    \right)
  }
where \eqn{Y_{i, t}}, \eqn{\eta_{0_{i, t}}},
\eqn{\eta_{1_{i, t}}},
and \eqn{\boldsymbol{\varepsilon}_{i, t}}
are random variables and
\eqn{\theta} is a model parameter.
\eqn{Y_{i, t}} is the observed random variable
at time \eqn{t} and individual \eqn{i},
\eqn{\eta_{0_{i, t}}} (intercept)
and
\eqn{\eta_{1_{i, t}}} (slope)
form a vector of latent random variables
at time \eqn{t} and individual \eqn{i},
and \eqn{\boldsymbol{\varepsilon}_{i, t}}
a vector of random measurement errors
at time \eqn{t} and individual \eqn{i}.
\eqn{\theta} is the variance of
\eqn{\boldsymbol{\varepsilon}}.

The dynamic structure is given by
\deqn{
    \left(
    \begin{array}{c}
    \eta_{0_{i, t}} \\
    \eta_{1_{i, t}} \\
    \end{array}
    \right)
    =
    \left(
    \begin{array}{cc}
      1 & 1 \\
      0 & 1 \\
    \end{array}
    \right)
    \left(
    \begin{array}{c}
    \eta_{0_{i, t - 1}} \\
    \eta_{1_{i, t - 1}} \\
    \end{array}
    \right) .
  }

The mean vector and covariance matrix of the intercept and slope
are captured in the mean vector and covariance matrix
of the initial condition given by
\deqn{
    \boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}
    =
    \left(
    \begin{array}{c}
      \mu_{\eta_{0}} \\
      \mu_{\eta_{1}} \\
    \end{array}
    \right) \quad \mathrm{and,}
  }

\deqn{
    \boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}
    =
    \left(
    \begin{array}{cc}
      \sigma^{2}_{\eta_{0}} &
      \sigma_{\eta_{0}, \eta_{1}} \\
      \sigma_{\eta_{1}, \eta_{0}} &
      \sigma^{2}_{\eta_{1}} \\
    \end{array}
    \right) .
  }
}

\subsection{Type 1}{

The measurement model is given by
\deqn{
    Y_{i, t}
    =
    \left(
    \begin{array}{cc}
      1 & 0 \\
    \end{array}
    \right)
    \left(
    \begin{array}{c}
      \eta_{0_{i, t}} \\
      \eta_{1_{i, t}} \\
    \end{array}
    \right)
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    0,
    \theta
    \right) .
  }

The dynamic structure is given by
\deqn{
    \left(
    \begin{array}{c}
    \eta_{0_{i, t}} \\
    \eta_{1_{i, t}} \\
    \end{array}
    \right)
    =
    \left(
    \begin{array}{cc}
      1 & 1 \\
      0 & 1 \\
    \end{array}
    \right)
    \left(
    \begin{array}{c}
    \eta_{0_{i, t - 1}} \\
    \eta_{1_{i, t - 1}} \\
    \end{array}
    \right)
    +
    \boldsymbol{\Gamma}
    \mathbf{x}_{i, t}
  }
where
\eqn{\mathbf{x}_{i, t}} represents a vector of covariates
at time \eqn{t} and individual \eqn{i},
and \eqn{\boldsymbol{\Gamma}} the coefficient matrix
linking the covariates to the latent variables.
}

\subsection{Type 2}{

The measurement model is given by
\deqn{
    Y_{i, t}
    =
    \left(
    \begin{array}{cc}
      1 & 0 \\
    \end{array}
    \right)
    \left(
    \begin{array}{c}
      \eta_{0_{i, t}} \\
      \eta_{1_{i, t}} \\
    \end{array}
    \right)
    +
    \boldsymbol{\kappa}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    0,
    \theta
    \right)
  }
where
\eqn{\boldsymbol{\kappa}} represents the coefficient matrix
linking the covariates to the observed variables.

The dynamic structure is given by
\deqn{
    \left(
    \begin{array}{c}
    \eta_{0_{i, t}} \\
    \eta_{1_{i, t}} \\
    \end{array}
    \right)
    =
    \left(
    \begin{array}{cc}
      1 & 1 \\
      0 & 1 \\
    \end{array}
    \right)
    \left(
    \begin{array}{c}
    \eta_{0_{i, t - 1}} \\
    \eta_{1_{i, t - 1}} \\
    \end{array}
    \right)
    +
    \boldsymbol{\Gamma}
    \mathbf{x}_{i, t} .
  }
}
}
\examples{
# prepare parameters
set.seed(42)
## number of individuals
n <- 5
## time points
time <- 5
## dynamic structure
p <- 2
mu0 <- c(0.615, 1.006)
sigma0 <- matrix(
  data = c(
    1.932,
    0.618,
    0.618,
    0.587
  ),
  nrow = p
)
sigma0_l <- t(chol(sigma0))
## measurement model
k <- 1
theta <- 0.50
theta_l <- sqrt(theta)
## covariates
j <- 2
x <- lapply(
  X = seq_len(n),
  FUN = function(i) {
    return(
      matrix(
        data = rnorm(n = j * time),
        nrow = j
      )
    )
  }
)
gamma <- diag(x = 0.10, nrow = p, ncol = j)
kappa <- diag(x = 0.10, nrow = k, ncol = j)

# Type 0
ssm <- SimSSMLinGrowth(
  n = n,
  time = time,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  theta_l = theta_l,
  type = 0
)

plot(ssm)

# Type 1
ssm <- SimSSMLinGrowth(
  n = n,
  time = time,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  theta_l = theta_l,
  type = 1,
  x = x,
  gamma = gamma
)

plot(ssm)

# Type 2
ssm <- SimSSMLinGrowth(
  n = n,
  time = time,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  theta_l = theta_l,
  type = 2,
  x = x,
  gamma = gamma,
  kappa = kappa
)

plot(ssm)

}
\references{
Chow, S.-M., Ho, M. R., Hamaker, E. L., & Dolan, C. V. (2010).
Equivalence and differences between structural equation modeling
and state-space modeling techniques.
\emph{Structural Equation Modeling: A Multidisciplinary Journal},
17(2), 303–332.
\doi{10.1080/10705511003661553}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{LinSDE2SSM}()},
\code{\link{PBSSMFixed}()},
\code{\link{PBSSMLinSDEFixed}()},
\code{\link{PBSSMOUFixed}()},
\code{\link{PBSSMVARFixed}()},
\code{\link{SimBetaN}()},
\code{\link{SimPhiN}()},
\code{\link{SimSSMFixed}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinSDEFixed}()},
\code{\link{SimSSMLinSDEIVary}()},
\code{\link{SimSSMOUFixed}()},
\code{\link{SimSSMOUIVary}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARIVary}()},
\code{\link{TestPhi}()},
\code{\link{TestStability}()},
\code{\link{TestStationarity}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{growth}
\keyword{sim}
\keyword{simStateSpace}
