%% File Name: f1d.irt.Rd
%% File Version: 1.21
%% File Last Change: 2017-07-11 16:26:56

\name{f1d.irt}
\alias{f1d.irt}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Functional Unidimensional Item Response Model
}
\description{
Estimates the functional unidimensional item response model for
dichotomous data (Ip et al., 2013). Either the IRT model is estimated
using a probit link and employing tetrachoric correlations or
item discriminations and intercepts of a pre-estimated multidimensional IRT
model are provided as input.
}
\usage{
f1d.irt(dat = NULL, nnormal = 1000, nfactors = 3, A = NULL, intercept = NULL, 
    mu = NULL , Sigma = NULL , maxiter = 100, conv = 10^(-5), progress = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{dat}{
Data frame with dichotomous item responses
}
  \item{nnormal}{
Number of \eqn{\theta_p} grid points for approximating the
normal distribution
}
  \item{nfactors}{
Number of dimensions to be estimated
}
  \item{A}{
Matrix of item discrminations (if the IRT model is already estimated) 
}
  \item{intercept}{
Vector of item intercepts (if the IRT model is already estimated)
}
\item{mu}{Vector of estimated means. In the default it is assumed
	 that all means are zero.}
\item{Sigma}{Estimated covariance matrix. In the default it is
the identity matrix.}
  \item{maxiter}{
Maximum number of iterations
}
  \item{conv}{
Convergence criterion
}
  \item{progress}{
Display progress? The default is \code{TRUE}.
}
}
\details{
The functional unidimensional item response model (F1D model) 
for dichotomous item responses is based on a multidimensional model with a
link function \eqn{g} (probit or logit):
\deqn{ P( X_{pi}=1 | \bold{\theta}_p ) = 
g( \sum_d a_{id} \theta_{pd} - d_i ) }
It is assumed that \eqn{\bold{\theta}_p} is multivariate normally
distribution with a zero mean vector and identity covariance matrix.

The F1D model estimates unidimensional item response functions
such that
\deqn{ P( X_{pi}=1 | \theta_p^\ast ) \approx 
g \left( a_{i}^\ast \theta_{p}^\ast - d_i^\ast \right) }
The optimization function \eqn{F} minimizes the deviations of
the approximation equations
\deqn{ a_{i}^\ast \theta_{p}^\ast - d_i^\ast 
\approx \sum_d a_{id} \theta_{pd} - d_i }
The optimization function \eqn{F} is defined by
\deqn{ F( \{ a_i^\ast , d_i^\ast \}_i , \{ \theta_p^\ast \}_p )= 	
	\sum_p \sum_i w_p ( a_{id} \theta_{pd} - d_i- 
a_{i}^\ast \theta_{p}^\ast + d_i^\ast )^2 \rightarrow Min! }
All items \eqn{i} are equally weighted whereas the ability
distribution of persons \eqn{p} are weighted according to the
multivariate normal distribution (using weights \eqn{w_p}). 
The estimation is conducted using an alternating least squares algorithm 
(see Ip et al. 2013 for a different algorithm). The ability distribution 
\eqn{\theta_p^\ast} of the functional unidimensional model is assumed 
to be standardized, i.e. does have a zero mean and a standard deviation of one.
}

\value{
A list with following entries:
\item{item}{Data frame with estimated item parameters: Item intercepts
    for the functional unidimensional \eqn{a_{i}^\ast} (\code{ai.ast}) and
    the ('ordinary') unidimensional (\code{ai0}) item response
    model. The same holds for item intercepts \eqn{d_{i}^\ast} (\code{di.ast} and
    \code{di0} respectively).
}
\item{person}{Data frame with estimated \eqn{\theta_p^\ast}
  distribution. Locations are \code{theta.ast} with corresponding
  probabilities in \code{wgt}.}
\item{A}{Estimated or provided item discriminations}
\item{intercept}{Estimated or provided intercepts}
\item{dat}{Used dataset}
\item{tetra}{Object generated by \code{\link{tetrachoric2}} if \code{dat}
	is specified as input. This list entry is useful for applying
	\code{\link{greenyang.reliability}}.
}
}


\references{
Ip, E. H., Molenberghs, G., Chen, S. H., Goegebeur, Y., & 
De Boeck, P. (2013). Functionally unidimensional item 
response models for multivariate binary data. 
\emph{Multivariate Behavioral Research}, \bold{48}, 534-562.
}

\author{
Alexander Robitzsch
}

%\note{
%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
For estimation of bifactor models and Green-Yang reliability
based on tetrachoric correlations see \code{\link{greenyang.reliability}}.

For estimation of bifactor models based on marginal maximum likelihood
(i.e. full information maximum likelihood) see the 
\code{\link[TAM:tam.fa]{TAM::tam.fa}} function in the \pkg{TAM} package.
}


\examples{              
#############################################################################
# EXAMPLE 1: Dataset Mathematics data.math | Exploratory multidimensional model
#############################################################################
data(data.math)
dat <- ( data.math$data )[ , -c(1,2) ] # select Mathematics items

#****
# Model 1: Functional unidimensional model based on original data

#++ (1) estimate model with 3 factors
mod1 <- sirt::f1d.irt( dat =dat , nfactors=3)

#++ (2) plot results
     par(mfrow=c(1,2))
# Intercepts
plot( mod1$item$di0 , mod1$item$di.ast , pch=16 , main="Item Intercepts" ,
        xlab= expression( paste( d[i] , " (Unidimensional Model)" )) ,
        ylab= expression( paste( d[i] , " (Functional Unidimensional Model)" )))
abline( lm(mod1$item$di.ast ~ mod1$item$di0) , col=2 , lty=2 )
# Discriminations
plot( mod1$item$ai0 , mod1$item$ai.ast , pch=16 , main="Item Discriminations" ,
        xlab= expression( paste( a[i] , " (Unidimensional Model)" )) ,
        ylab= expression( paste( a[i] , " (Functional Unidimensional Model)" )))
abline( lm(mod1$item$ai.ast ~ mod1$item$ai0) , col=2 , lty=2 )
     par(mfrow=c(1,1))

#++ (3) estimate bifactor model and Green-Yang reliability
gy1 <- sirt::greenyang.reliability( mod1$tetra , nfactors = 3 )

\dontrun{
#****
# Model 2: Functional unidimensional model based on estimated multidimensional
#          item response model

#++ (1) estimate 2-dimensional exploratory factor analysis with 'smirt'
I <- ncol(dat)
Q <- matrix( 1, I,2 )
Q[1,2] <- 0
variance.fixed <- cbind( 1,2,0 )
mod2a <- sirt::smirt( dat , Qmatrix=Q , irtmodel="comp" , est.a="2PL" , 
                variance.fixed=variance.fixed , maxiter=50)
#++ (2) input estimated discriminations and intercepts for
#       functional unidimensional model
mod2b <- sirt::f1d.irt( A = mod2a$a , intercept = mod2a$b )

#############################################################################
# EXAMPLE 2: Dataset Mathematics data.math | Confirmatory multidimensional model
#############################################################################

data(data.math)
library(TAM)

# dataset
dat <- data.math$data
dat <- dat[ , grep("M" , colnames(dat) ) ]
# extract item informations
iteminfo <- data.math$item
I <- ncol(dat)
# define Q-matrix
Q <- matrix( 0 , nrow=I , ncol=3 )
Q[ grep( "arith" , iteminfo$domain ) , 1 ] <- 1
Q[ grep( "Meas" , iteminfo$domain ) , 2 ] <- 1
Q[ grep( "geom" , iteminfo$domain ) , 3 ] <- 1

# fit three-dimensional model in TAM
mod1 <- TAM::tam.mml.2pl(  dat , Q=Q , control=list(maxiter=40 , snodes=1000) )
summary(mod1)

# specify functional unidimensional model
intercept <- mod1$xsi[ , c("xsi") ]
names(intercept) <- rownames(mod1$xsi)
fumod1 <- sirt::f1d.irt( A = mod1$B[,2,] , intercept=intercept , Sigma= mod1$variance)
fumod1$item
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{Functional unidimensional item response model}
%% \keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
