% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/sheppInt.R
\name{qsheppInt}
\alias{qsheppInt}
\alias{qsheppInt2d}
\alias{qsheppInt3d}
\title{Shepard's (modified) quadratic interpolation method}
\usage{
qsheppInt(X, y, XNew = NULL, nQ = 15L, nW = 33L, nR,
          checkX = TRUE)
qsheppInt2d(X, y, XNew = NULL, nQ = 13L, nW = 19L, nR,
            deriv = 0L, checkX = TRUE)
qsheppInt3d(X, y, XNew = NULL, nQ = 17L, nW = 32L, nR,
            deriv = 0L, checkX = TRUE)
}
\arguments{
\item{X}{Design matrix. A matrix with \eqn{n} rows and \eqn{d}
columns where \eqn{n} is the number of nodes and \eqn{d} is the
dimension.}

\item{y}{Numeric vector of length \eqn{n} containing the function
values at design points.}

\item{XNew}{Numeric matrix of new design points where
interpolation is computed. It must have \eqn{d} columns.}

\item{nQ}{Number of points used in the local polynomial fit. This
is the parameter \code{NQ} of the Fortran routine \code{qshepmd}
and the parameter \eqn{N_p} of the referenced article.}

\item{nW}{Number of nodes within (and defining) the radii of
influence, which enter into the weights. This is parameter
\code{NW} of the Fortran routine \code{qshepmd} and \eqn{N_w} of
the reference article.}

\item{nR}{Number of divisions in each dimension for the cell grid
defined in the subroutine \code{storem}.  A hyperbox containing
the nodes is partitioned into cells in order to increase search
efficiency.  The recommended value \eqn{(n/3)^(1/d)} is used as
default, where \eqn{n} is the number of nodes and \eqn{d} is the
dimension.}

\item{checkX}{If \code{TRUE}, the dimensions and colnames of
\code{X} and \code{XNew} will be checked.}

\item{deriv}{Logical or integer value in \code{c(0, 1)}. When the
(coerced) integer value is \code{1}, the derivatives are computed.
This is is only possible for the dimensions \code{2} and \code{3},
and only when the specific interpolation functions are used.
The result is then a matrix with one column for the function and
one column by derivative.}
}
\value{
A list with several objects related to the computation
method. The vector of interpolated value is in the list element
named \code{yNew}.
}
\description{
Shepard's (modified) quadratic interpolation method.
}
\details{
Shepard's modified interpolation method works for scattered data
in arbitrary dimension. It relies on a \emph{local} polynomial fit
and the quadratic version uses a polynomial of degree 2 (a quadratic
form) as local approximation of the function.
}
\note{
This function is an R interface to the \code{qshepmd}
routine in the \pkg{SHEPPACK} Fortran package available on netlib
\url{http://www.netlib.org} as algorithm 905A.

The \code{qshepInt} function is an interface for the
\code{QSHEPMD} Fortran routine, while \code{qshepInt2d} and
\code{qshepInt3d} are interfaces to the \code{QSHEPM2D} and
\code{QSHEPM3D} Fortran routines. The general interpolation of
\code{qshepInt} can be used also for the dimensions \eqn{2} and
\eqn{3}. However, this function does not allow the computation of
the derivatives as \code{qshepInt2d} and \code{qshepInt3d} do.
}
\examples{
n <- 1500; nNew <- 100; d <- 4
fTest <- function(x)((x[1] + 2 * x[2]   + 3 * x[3] + 4 * x[4]) / 12)^2
set.seed(12345)
X <- matrix(runif(n*d), nrow = n, ncol = d)
y <- apply(X, 1, FUN = fTest)
XNew <- matrix(runif(nNew * d), nrow = nNew, ncol = d)
yNew <- apply(XNew, 1, FUN = fTest)
system.time(res <- qsheppInt(X = X, XNew = XNew, y = y, nQ = 40,
                             checkX = FALSE))
## check errors
max(abs(res$yNew - yNew))

##=========================================================================
## Use SHEPPACK test functions see Thacker et al. section 7 'PERFORMANCE'
##=========================================================================
\dontrun{
   set.seed(1234)
   d <- 3
   k <- 0:4; n0 <- 100 * 2^k; n1 <- 4
   GD <- Grid(nlevels = rep(n1, d))
   XNew <- as.matrix(GD)
   RMSE <- array(NA, dim = c(5, length(k)),
                 dimnames = list(fun = 1:5, k = k))
   for (iFun in 1:5) {
      yNew <- apply(XNew, 1, ShepFuns[[iFun]])
      for (iN in 1:length(n0)) {
         X <- matrix(runif(n0[iN] * d), ncol = d)
         y <- apply(X, 1, ShepFuns[[iFun]])
         res <- qsheppInt(X = X, XNew = XNew, y = y, nQ = 40, checkX = FALSE)
         RMSE[iFun, iN] <- mean((res$yNew - yNew)^2)
      }
   }
   cols <- c("black", "SteelBlue2", "orangered", "SpringGreen3", "purple")
   pchs <-  c(16, 21, 22, 23, 24)
   matplot(k, t(RMSE), type = "o", lwd = 2, lty = 1,
           col = cols, xaxt = "n", pch = pchs, cex = 1.4,
           bg = "white",
           main = sprintf("dim = \%d SHEPPACK test functions", d),
           xlab = "number of nodes", ylab = "RMSE")
   axis(side = 1, at = k, labels = n0)
   legend("topright", legend = paste("shepFun", 1:5),
           col = cols, pch = pchs, lwd = 2, pt.lwd = 2, pt.bg = "white")
}
}
\author{
Fortran code by William I. Thacker; Jingwei
Zhang; Layne T. Watson; Jeffrey B. Birch; Manjula A. Iyer; Michael
W. Berry. See References below.  The Fortran code is a translation
of M.W Berry's C++ code.

Code adaptation and R interface by Yves Deville.
}
\references{
W.I. Thacker, J. Zhang, L.T. Watson, J.B. Birch,
M.A. Iyer and M.W. Berry (2010). Algorithm 905: SHEPPACK: Modified
Shepard Algorithm for Interpolation of Scattered Multivariate Data
\emph{ACM Trans. on Math. Software} (TOMS) Vol. 37, n. 3.
\href{http://dl.acm.org/citation.cfm?id=1824812}{link}

M.W. Berry and K.S. Minser (1999). Algorithm 798: High-dimensional
interpolation using the modified Shepard method. \emph{ACM
Trans. Math. Software} (TOMS) Vol. 25, n. 3, pp. 353-366.
\href{http://dl.acm.org/citation.cfm?id=326147.326154}{link}
}

