% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elfunctions.R
\name{EL0}
\alias{EL0}
\title{Uni-variate empirical likelihood via direct lambda search}
\usage{
EL0(
  z,
  mu = NULL,
  ct = NULL,
  shift = NULL,
  renormalise = FALSE,
  return.weights = FALSE,
  weight.tolerance = NULL,
  boundary.tolerance = 1e-09,
  trunc.to = 0,
  deriv = FALSE,
  log.control = list(order = NULL, lower = NULL, upper = NULL),
  verbose = FALSE
)
}
\arguments{
\item{z}{A numeric vector containing the observations.}

\item{mu}{Hypothesised mean of \code{z} in the moment condition.}

\item{ct}{Numeric count variable with non-negative values that indicates the multiplicity of observations.
Can be fractional. Very small counts below the threshold \code{weight.tolerance} are zeroed.}

\item{shift}{The value to add in the denominator (useful in case there are extra Lagrange multipliers):
\eqn{1 + \lambda'Z + shift}{1 + lambda'Z + shift.}.}

\item{renormalise}{If \code{FALSE}, then uses the total sum of counts as the number of observations,
like in vanilla empirical likelihood, due to formula (2.9) in \insertCite{owen2001empirical}{smoothemplik},
otherwise re-normalises the counts to 1 according to \insertCite{cosma2019inference}{smoothemplik}
(see p. 170, the topmost formula).}

\item{return.weights}{Logical: if \code{TRUE}, returns the empirical probabilities.
Default is memory-saving (\code{FALSE}).}

\item{weight.tolerance}{Weight tolerance for counts to improve numerical stability
(defaults to \code{sqrt(.Machine$double.eps)} times the maximum weight).}

\item{boundary.tolerance}{Relative tolerance for determining when lambda is not an interior
solution because it is too close to the boundary. Unit: fraction of the feasble bracket length.}

\item{trunc.to}{Counts under \code{weight.tolerance} will be set to this value.
In most cases, setting this to \code{0} (default) or \code{weight.tolerance}
is a viable solution for the zero-denominator problem.}

\item{deriv}{Logical: if \code{TRUE}, computes and returns the first two derivatives of log-ELR w.r.t. \code{mu}.}

\item{log.control}{List of arguments passed to [logTaylor()].}

\item{verbose}{Logical: if \code{TRUE}, prints warnings.}
}
\value{
A list with the following elements:

\describe{
  \item{logelr}{Logarithm of the empirical likelihood ratio.}
  \item{lam}{The Lagrange multiplier.}
  \item{wts}{Observation weights/probabilities (of the same length as \code{z}).}
  \item{converged}{\code{TRUE} if the algorithm converged, \code{FALSE} otherwise (usually means that \code{mu} is not within the range of \code{z}, i.e. the one-dimensional convex hull of \code{z}).}
  \item{iter}{The number of iterations used (from \code{brentZero}).}
  \item{bracket}{The admissible interval for lambda (that is, yielding weights between 0 and 1).}
  \item{estim.prec}{The approximate estimated precision of lambda (from \code{brentZero}).}
  \item{f.root}{The value of the derivative of the objective function w.r.t. lambda at the root (from \code{brentZero}). Values \code{> sqrt(.Machine$double.eps)} indicate convergence problems.}
  \item{deriv}{If requested, the first two derivatives of log-ELR w.r.t. \code{mu}}
  \item{exitcode}{An integer indicating the reason of termination.}
  \item{message}{Character string describing the optimisation termination status.}
}
}
\description{
Empirical likelihood with counts to solve one-dimensional problems efficiently with Brent's root search algorithm.
Conducts an empirical likelihood ratio test of the hypothesis that the mean of \code{z} is \code{mu}.
The names of the elements in the returned list are consistent with the original R code
in \insertCite{owen2017weighted}{smoothemplik}.
}
\details{
This function provides the core functionality for univariate empirical likelihood.
The technical details is given in \insertCite{cosma2019inference}{smoothemplik},
although the algorithm used in that paper is slower than the one provided by this function.

Since we know that the EL probabilities belong to (0, 1), the interval (bracket) for \eqn{\lambda}{l} search
can be determined in the spirit of formula (2.9) from \insertCite{owen2001empirical}{smoothemplik}. Let
\eqn{z^*_i := z_i - \mu}{Z[i] := z[i] - mu} be the recentred observations.
\deqn{p_i = c_i/N \cdot (1 + \lambda z^*_i + s)^{-1}}{p[i] = c[i]/N * 1/(1 + l*Z[i] + s)}
The probabilities are bounded from above: \eqn{p_i<1}{p[i] < 1} for all \emph{i}, therefore,
\deqn{c_i/N \cdot (1 + \lambda z^*_i + s)^{-1} < 1}{c[i]/N * 1/(1 + l*Z[i] + s) < 1}
\deqn{c_i/N - 1 - s < \lambda z^*_i}{c[i]/N - 1 - s < l*Z[i]}
Two cases: either \eqn{z^*_i<0}{Z[i] < 0}, or \eqn{z^*_i>0}{Z[i] > 0}
(cases with \eqn{z^*_i=0}{Z[i] = 0} are trivially excluded because they do not affect the EL). Then,
\deqn{(c_i/N - 1 - s)/z^*_i > \lambda,\ \forall i: z^*_i<0}{(c[i]/N - 1 - s)/Z[i] > l,  such i that Z[i]<0}
\deqn{(c_i/N - 1 - s)/z^*_i < \lambda,\ \forall i: z^*_i>0}{(c[i]/N - 1 - s)/Z[i] < l,  such i that Z[i]>0}
which defines the search bracket:
\deqn{\lambda_{\min} := \max_{i: z^*_i>0} (c_i/N - 1 - s)/z^*_i}{l > max_{i: Z[i]>0} (c_i/N - 1 - s)/Z[i]}
\deqn{\lambda_{\max} := \min_{i: z^*_i<0} (c_i/N - 1 - s)/z^*_i}{l < min_{i: Z[i]<0} (c_i/N - 1 - s)/Z[i]}
\deqn{\lambda_{\min} < \lambda < \lambda_{\max}}

(This derivation contains \emph{s}, which is the extra shift that extends the
function to allow mixed conditional and unconditional estimation;
Owen's textbook formula corresponds to \eqn{s = 0}{s = 0}.)

The actual tolerance of the lambda search in \code{brentZero} is
\eqn{2 |\lambda_{\max}| \epsilon_m + \mathrm{tol}/2}{2 * MachEps * l_max + tol/2},
where \code{tol} is \code{.Machine$double.eps} and
\eqn{\epsilon_m}{MachEps} is \code{.Machine$double.eps}.

The sum of log-weights is maximised without Taylor expansion, forcing \code{mu} to be inside
the convex hull of \code{z}. If a violation is happening, consider using
\code{log.control(order = 4)} or switching to Euclidean likelihood via [EuL()].
}
\examples{
# Figure 2.4 from Owen (2001) -- with a slightly different data point
earth <- c(
  5.5, 5.61, 4.88, 5.07, 5.26, 5.55, 5.36, 5.29, 5.58, 5.65, 5.57, 5.53, 5.62, 5.29,
  5.44, 5.34, 5.79, 5.1, 5.27, 5.39, 5.42, 5.47, 5.63, 5.34, 5.46, 5.3, 5.75, 5.68, 5.85
)
# Root searching (EL0) is faster than minimisation w.r.t. lambda (EL1)
set.seed(1)
system.time(r0 <- replicate(40, EL0(sample(earth, replace = TRUE), mu = 5.517)))
set.seed(1)
system.time(r1 <- replicate(40, EL1(sample(earth, replace = TRUE), mu = 5.517)))
plot(apply(r0, 2, "[[", "logelr"), apply(r1, 2, "[[", "logelr") - apply(r0, 2, "[[", "logelr"),
     bty = "n", xlab = "log(ELR) computed via dampened Newthon method",
     main = "Discrepancy between EL1 and EL0", ylab = "")
abline(h = 0, lty = 2)

# Handling the convex hull violation differently
EL0(1:9)
EL0(1:9, log.control = list(order = 2))  # Warning + huge lambda
EL0(1:9, log.control = list(order = 4))  # Warning + huge lambda

# Warning: depending on the compiler, the discrepancy between EL and EL0
# can be one million (1) times larger than the machine epsilon despite both of them
# being written in pure R
# The results from Apple clang-1400.0.29.202 and Fortran GCC 12.2.0 are different from
# those obtained under Ubuntu 22.04.4 + GCC 11.4.0-1ubuntu1~22.04,
# Arch Linux 6.6.21 + GCC 14.1.1, and Windows Server 2022 + GCC 13.2.0
out0 <- EL0(earth, mu = 5.517, return.weights = TRUE)[1:4]
out1 <- EL1(earth, mu = 5.517, return.weights = TRUE)[1:4]
print(c(out0$lam, out1$lam), 16)

# Value of lambda                                EL0                 EL1
# aarch64-apple-darwin20          -1.5631313957?????  -1.5631313955?????
# Windows, Ubuntu, Arch           -1.563131395492627  -1.563131395492627
}
\references{
\insertAllCited{}
}
\seealso{
[EL1()] for multi-variate EL based on minimisation w.r.t. lambda.
}
