% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sk_var.R
\name{sk_toep_mult}
\alias{sk_toep_mult}
\title{Efficiently compute yzx for symmetric Toeplitz matrices y and x}
\usage{
sk_toep_mult(y, z = NULL, x = NULL, idx_obs = NULL, gdim = NULL)
}
\arguments{
\item{y}{numeric matrix or vector, the symmetric Toeplitz matrix y or its first row}

\item{z}{numeric matrix or vector with dimensionality conforming with y (and x)}

\item{x}{numeric matrix or vector, the symmetric Toeplitz matrix x or its first row}

\item{idx_obs}{integer vector, indices of the observed grid points}
}
\value{
numeric matrix, the product of yzx or yz (if x is NULL)
}
\description{
Computes the product \code{y \%*\% z} or \code{y \%*\% z \%*\% x} for symmetric Toeplitz matrices
\code{y} and \code{x} and any numeric matrix \code{z}.
}
\details{
Argument(s) \code{y} (and \code{x}) can be vector(s) supplying the first row of the matrix.
By default, \code{z} is the identity matrix, so for matrix \code{y}, sk_toep_mult(\code{y}) returns
its argument, and for vector \code{y}, it returns the Toeplitz matrix generated by \code{y}, the
same as \code{stats::toeplitz(y)}.

Fast Fourier transforms are used to reduce the memory footprint of computations,
The first row(s) of \code{y} (and \code{x}) are embedded in a zero-padded vector representing a
circulant matrix, whose action on the zero-padded version of \code{z} is equivalent to
element-wise product in Fourier space. This allows the desired matrix product to be
computed without explicitly creating matrices \code{y} or \code{x} in memory.

The function is optimized for grid data \code{z} that are sparse (many zeros). Before
computing any transformations it first scans for and removes columns and rows of
z which are all zero, replacing them afterwards.

To avoid unnecessarily copying large sparse matrices, \code{z} can be the vector of
non-zero matrix entries only, where \code{gdim} specifies the full matrix dimensions and
\code{idx_obs} the indices of the non-zero entries.
}
\examples{
# define example matrix from 1D exponential variogram
n = 10
y = exp(1-seq(n))
y_mat = sk_toep_mult(y)
max( abs(y_mat - stats::toeplitz(y))  )

# multiply by random matrix and compare with default matrix multiply
z = matrix(stats::rnorm(n^2), n)
result_default = y_mat \%*\% z
max( abs( result_default - sk_toep_mult(y_mat, z) ) )

# save memory by passing only the first row of the Toeplitz matrix
max( abs( result_default - sk_toep_mult(y, z) ) )

# sparsify z and repeat
idx_sparse = sample.int(n^2, n^2 - n)
z[idx_sparse] = 0
result_default = y_mat \%*\% z
max( abs( result_default - sk_toep_mult(y, z) ) )

# right-multiply with another kernel
x = exp( 2 *( 1-seq(n) ) )
x_mat = sk_toep_mult(x)
result_default = result_default \%*\% x_mat
max( abs( result_default - sk_toep_mult(y, z, x) ) )

# z can also be supplied as vector of nonzero grid values
idx_obs = which(z != 0)
gdim = c(y=n, x=n)
max( abs( result_default - sk_toep_mult(y, z=z[idx_obs], x, idx_obs, gdim) ) )

}
\seealso{
base::toeplitz stats::fft

Other internal variance-related functions: 
\code{\link{sk_corr_mat}()},
\code{\link{sk_corr}()},
\code{\link{sk_var_mult}()}
}
\concept{internal variance-related functions}
\keyword{internal}
