% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/raster.change.R
\name{raster.change}
\alias{raster.change}
\title{Raster change between two nominal rasters}
\usage{
raster.change(
  x,
  y,
  d = c(3, 3),
  stat = c("kappa", "wkappa", "t.test", "cor", "entropy", "cross-entropy",
    "divergence"),
  w = NULL,
  out.raster = NULL,
  mask = FALSE,
  force.memory = FALSE
)
}
\arguments{
\item{x}{First raster for comparison, rasterLayer class object}

\item{y}{Second raster for comparison, rasterLayer class object}

\item{d}{Rectangular window size, must be odd but not necessarily 
square}

\item{stat}{Statistic to use in comparison, please see details for 
options.}

\item{w}{Weights if stat="kappa", must represent same classes as 
input rasters}

\item{out.raster}{Optional output raster}

\item{mask}{(FALSE/TRUE) mask output to original rasters}

\item{force.memory}{(FALSE/TRUE) Force in memory processing, may fail with 
insufficient RAM}
}
\value{
A raster layer or stack object one of the following layers:
\itemize{ 
\item   kappa         Kappa or Weighted Kappa statistic (if stat = "kappa")
\item   correlation   Paired t.test statistic  (if stat = "cor")
\item   entropy       Delta entropy  (if stat = "entropy")
\item   divergence    Kullback-Leibler divergence (if stat = "divergence")
\item   cross.entropy Cross-entropy (if stat = "cross.entropy")
\item   t.test        Paired t.test statistic  (if stat = "t.test")
\item   p.value       p-value of the paired t.test statistic (if stat = "t.test")
 }
}
\description{
Compares two categorical rasters with a variety of 
             statistical options

This function provides a various statistics for comparing two classified maps. 
Valid options are:
\itemize{ 
\item   kappa - Cohen's Kappa 
\item   wkappa - Cohen's Weighted Kappa (not yet implemented) 
\item   t.test - Two-tailed paired t-test 
\item   cor - Persons Correlation 
\item   entropy - Delta entropy 
\item   cross-entropy - Cross-entropy loss function 
\item   divergence - Kullback-Leibler divergence (relative entropy) 
}

Kappa and t-test values < 0 are reported as 0. For a weighted kappa, a matrix must 
be provided that correspond to the pairwise weights for all values in both rasters. 
Delta entropy is derived by calculating Shannon's on each focal window then 
differencing  them (e(x) - e(y))
}
\examples{
\donttest{
 library(sp)
 library(raster)
 data(meuse.grid)
 r1 <- sp::SpatialPixelsDataFrame(points = meuse.grid[c("x", "y")], 
                                  data = meuse.grid)
r1 <- raster(r1)
  na.idx <- which(!is.na(r1[]))
  r1[na.idx] <- round(runif(length(na.idx), 1,5),0)
 r2 <- sp::SpatialPixelsDataFrame(points = meuse.grid[c("x", "y")], 
                                  data = meuse.grid) 
r2 <- raster(r2)
  r2[na.idx] <- round(runif(length(na.idx), 1,5),0)
  
s = 11    
 ( r.kappa <- raster.change(r1, r2, d = s, mask = TRUE) )   
 ( r.ttest <- raster.change(r1, r2, d = s, stat="t.test", mask = TRUE) )
 ( r.ent <- raster.change(r1, r2, d = s, stat="entropy", mask = TRUE) )   
 ( r.cor <- raster.change(r1, r2, d = s, stat="cor", mask = TRUE) )
 ( r.ce <- raster.change(r1, r2, d = s, stat = "cross-entropy", mask = TRUE) )
 ( r.kl <- raster.change(r1, r2, d = s, stat = "divergence", mask = TRUE) )	
     
 opar <- par(no.readonly=TRUE)
 par(mfrow=c(3,2))
   plot(r.kappa, main="Kappa")
   plot(r.ttest[[1]], main="Paired t-test")
   plot(r.ent, main="Delta Entropy")
   plot(r.cor, main="Rank Correlation")
   plot(r.kl, main="Kullback-Leibler")
   plot(r.ce, main="cross-entropy")
 par(opar)

}

}
\references{
Cohen, J. (1960). A coefficient of agreement for nominal scales. Educational  
  and Psychological Measurement, 20:37-46

McHugh M.L. (2012) Interrater reliability: the kappa statistic. 
  Biochemia medica, 22(3):276–282.

Kullback, S., R.A. Leibler (1951). On information and sufficiency. Annals of 
   Mathematical Statistics. 22(1):79–86
}
\author{
Jeffrey S. Evans  <jeffrey_evans@tnc.org>
}
