% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmFilter.R, R/methods.R
\name{lmFilter}
\alias{lmFilter}
\alias{summary.spfilter}
\title{Unsupervised Spatial Filtering with Eigenvectors in Linear Regression Models}
\usage{
lmFilter(
  y,
  x = NULL,
  W,
  objfn = "MI",
  MX = NULL,
  sig = 0.05,
  bonferroni = TRUE,
  positive = TRUE,
  ideal.setsize = FALSE,
  alpha = 0.25,
  tol = 0.1,
  boot.MI = NULL,
  na.rm = TRUE
)

\method{summary}{spfilter}(object, EV = FALSE, ...)
}
\arguments{
\item{y}{response variable}

\item{x}{vector/ matrix of regressors (default = NULL)}

\item{W}{spatial connectivity matrix}

\item{objfn}{the objective function to be used for eigenvector selection.
Possible criteria are: the maximization of the adjusted R-squared ('R2'),
minimization of residual autocorrelation ('MI'), significance level of
candidate eigenvectors ('p'), significance of residual spatial
autocorrelation ('pMI') or all eigenvectors in the candidate set ('all')}

\item{MX}{covariates used to construct the projection matrix (default = NULL) - see
Details}

\item{sig}{significance level to be used for eigenvector selection
if \code{objfn = 'p'} or \code{objfn = 'pMI'}}

\item{bonferroni}{Bonferroni adjustment for the significance level
(TRUE/ FALSE) if \code{objfn = 'p'}. Set to FALSE if \code{objfn = 'pMI'} -
see Details}

\item{positive}{restrict search to eigenvectors associated with positive
levels of spatial autocorrelation (TRUE/ FALSE)}

\item{ideal.setsize}{if \code{positive = TRUE}, uses the formula proposed by
Chun et al. (2016) to determine the ideal size of the candidate set
(TRUE/ FALSE)}

\item{alpha}{a value in (0,1] indicating the range of candidate eigenvectors
according to their associated level of spatial autocorrelation, see e.g.,
Griffith (2003)}

\item{tol}{if \code{objfn = 'MI'}, determines the amount of remaining residual
autocorrelation at which the eigenvector selection terminates}

\item{boot.MI}{number of iterations used to estimate the variance of Moran's I.
If \code{boot.MI = NULL} (default), analytical results will be used}

\item{na.rm}{remove observations with missing values (TRUE/ FALSE)}

\item{object}{an object of class \code{spfilter}}

\item{EV}{display summary statistics for selected eigenvectors (TRUE/ FALSE)}

\item{...}{additional arguments}
}
\value{
An object of class \code{spfilter} containing the following
information:
\describe{
\item{\code{estimates}}{summary statistics of the parameter estimates}
\item{\code{varcovar}}{estimated variance-covariance matrix}
\item{\code{EV}}{a matrix containing the summary statistics of selected eigenvectors}
\item{\code{selvecs}}{vector/ matrix of selected eigenvectors}
\item{\code{evMI}}{Moran coefficient of all eigenvectors}
\item{\code{moran}}{residual autocorrelation in the initial and the
filtered model}
\item{\code{fit}}{adjusted R-squared of the initial and the filtered model}
\item{\code{residuals}}{initial and filtered model residuals}
\item{\code{other}}{a list providing supplementary information:
\describe{
\item{\code{ncandidates}}{number of candidate eigenvectors considered}
\item{\code{nev}}{number of selected eigenvectors}
\item{\code{sel_id}}{ID of selected eigenvectors}
\item{\code{sf}}{vector representing the spatial filter}
\item{\code{sfMI}}{Moran coefficient of the spatial filter}
\item{\code{model}}{type of the fitted regression model}
\item{\code{dependence}}{filtered for positive or negative spatial dependence}
\item{\code{objfn}}{selection criterion specified in the objective function of
the stepwise regression procedure}
\item{\code{bonferroni}}{TRUE/ FALSE: Bonferroni-adjusted significance level
(if \code{objfn = 'p'})}
\item{\code{siglevel}}{if \code{objfn = 'p'} or \code{objfn = 'pMI'}: actual
(unadjusted/ adjusted) significance level}
}
}
}
}
\description{
This function implements the eigenvector-based semiparametric
spatial filtering approach in a linear regression framework using ordinary least
squares (OLS). Eigenvectors are selected by an unsupervised stepwise regression
technique. Supported selection criteria are the minimization of residual
autocorrelation, maximization of model fit, significance of residual autocorrelation,
and the statistical significance of eigenvectors. Alternatively, all eigenvectors in
the candidate set can be included as well.
}
\details{
If \strong{W} is not symmetric, it gets symmetrized by
1/2 * (\strong{W} + \strong{W}') before the decomposition.

If covariates are supplied to \code{MX}, the function uses these regressors
to construct the following projection matrix:

\strong{M} = \strong{I} - \strong{X} (\strong{X}'\strong{X})^-1\strong{X}'

Eigenvectors from \strong{MWM} using this specification of
\strong{M} are not only mutually uncorrelated but also orthogonal
to the regressors specified in \code{MX}. Alternatively, if \code{MX = NULL}, the
projection matrix becomes \strong{M} = \strong{I} - \strong{11}'/*n*,
where \strong{1} is a vector of ones and *n* represents the number of
observations. Griffith and Tiefelsdorf (2007) show how the choice of the appropriate
\strong{M} depends on the underlying process that generates the spatial
dependence.

The Bonferroni correction is only possible if eigenvector selection is based on
the significance level of the eigenvectors (\code{objfn = 'p'}). It is set to
FALSE if eigenvectors are added to the model until the residuals exhibit no
significant level of spatial autocorrelation (\code{objfn = 'pMI'}).
}
\examples{
data(fakedata)
y <- fakedataset$x1
X <- cbind(fakedataset$x2, fakedataset$x3, fakedataset$x4)

res <- lmFilter(y = y, x = X, W = W, objfn = 'MI', positive = FALSE)
print(res)
summary(res, EV = TRUE)

E <- res$selvecs
(ols <- coef(lm(y ~ X + E)))
coef(res)

}
\references{
Tiefelsdorf, Michael and Daniel A. Griffith (2007):
Semiparametric filtering of spatial autocorrelation: the eigenvector
approach. Environment and Planning A: Economy and Space, 39 (5):
pp. 1193 - 1221.

Griffith, Daniel A. (2003): Spatial Autocorrelation and Spatial Filtering:
Gaining Understanding Through Theory and Scientific Visualization.
Berlin/ Heidelberg, Springer.

Chun, Yongwan, Daniel A. Griffith, Monghyeon Lee, Parmanand
Sinha (2016): Eigenvector selection with stepwise regression techniques
to construct eigenvector spatial filters. Journal of Geographical
Systems, 18, pp. 67 – 85.

Le Gallo, Julie and Antonio Páez (2013): Using synthetic
variables in instrumental variable estimation of spatial series models.
Environment and Planning A: Economy and Space, 45 (9): pp. 2227 - 2242.

Tiefelsdorf, Michael and Barry Boots (1995): The Exact Distribution
of Moran's I. Environment and Planning A: Economy and Space, 27 (6):
pp. 985 - 999.
}
\seealso{
\code{\link{glmFilter}}, \code{\link{getEVs}}, \code{\link{MI.resid}}
}
