% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_stelfi.r
\name{fit_stelfi}
\alias{fit_stelfi}
\title{Modelling spatiotemporal self-excitement}
\usage{
fit_stelfi(
  times,
  locs,
  sf,
  smesh,
  parameters,
  covariates,
  GMRF = FALSE,
  time_independent = TRUE,
  tmb_silent = TRUE,
  nlminb_silent = TRUE,
  ...
)
}
\arguments{
\item{times}{A vector of numeric observed time points.}

\item{locs}{A \code{data.frame} of \code{x} and \code{y} locations, 2xn.}

\item{sf}{An \code{sf} of type \code{POLYGON} specifying the spatial region
of the domain.}

\item{smesh}{A Delaunay triangulation of the spatial domain returned by \code{fmesher::fm_mesh_2d()}.}

\item{parameters}{A list of named parameters:
\itemize{
\item \code{coefs}, logged base rate of the Hawkes process and coefficients of covariates
\item \code{alpha}, intensity jump after an event occurrence
\item \code{beta},  rate of exponential decay of intensity after event occurrence
\item \code{tau},  \eqn{\tau} parameter for the GMRF (supplied only if \code{GMRF = TRUE})
\item \code{kappa}, \eqn{\kappa} parameter for the GMRF (supplied only if \code{GMRF = TRUE})
\item \code{xsigma}, standard deviation on x-axis of self-exciting kernel (if \code{time_independent = FALSE}, it is the s.d. after 1 time unit)
\item \code{ysigma}, standard deviation on y-axis of self-exciting kernel (if \code{time_independent = FALSE}, it is the s.d. after 1 time unit)
\item \code{rho}, correlation between x and y for the self-exciting kernel (the off-diagonal elements in the kernel's covariate matrix are \code{xsigma * ysigma * rho})
}}

\item{covariates}{Optional, a \code{matrix} of covariates at each
\code{smesh} node.}

\item{GMRF}{Logical, default \code{FALSE}. If \code{TRUE}, a Gaussian Markov
Random Field is included as a latent spatial effect.}

\item{time_independent}{Logical, default \code{TRUE}. If \code{FALSE}, Gaussian kernels have a
covariate matrix that is proportional to time since the event.
Warning, this is very memory intensive.}

\item{tmb_silent}{Logical, if \code{TRUE} (default) then
TMB inner optimisation tracing information will be printed.}

\item{nlminb_silent}{Logical, if \code{TRUE} (default) then for each iteration
\code{nlminb()} output will be printed.}

\item{...}{Additional arguments to pass to \code{optim()}}
}
\value{
A list containing components of the fitted model, see \code{TMB::MakeADFun}. Includes
\itemize{
\item \code{par}, a numeric vector of estimated parameter values;
\item \code{objective}, the objective function; and
\item \code{gr}, the TMB calculated gradient function.
}
}
\description{
Fits spatiotemporal Hawkes models. The self-excitement is 
Gaussian in space and exponentially decaying in time.
}
\details{
Temporal self-excitement follows an exponential decay function.
The self-excitement over space follows a Gaussian distribution centered at the triggering event.
There are two formulations of this model. The default is that the Gaussian function has a fixed spatial
covariance matrix, independent of time. Alternatively, covariance can be directly proportional to time,
meaning that the self-excitement radiates out from the center over time.
This can be appropriate when the mechanism causing self-excitement travels
at a finite speed, but is very memory-intensive. The spatiotemporal intensity function
used by \code{\link{stelfi}} is
\eqn{\lambda(s,t) = \mu + \alpha \Sigma_{i:\tau_i<t}(\textrm{exp}(-\beta * (t-\tau_i)) G_i(s-x_i, t - \tau_i))}
where
\itemize{
\item \eqn{\mu} is the background rate,
\item \eqn{\beta} is the rate of temporal decay,
\item \eqn{\alpha} is the increase in intensity after an event,
\item \eqn{\tau_i} are the event times,
\item \eqn{x_i} are the event locations (in 2D Euclidean space), and
\item \eqn{G_i(s-x_i, t - \tau_i)} is the spatial self-excitement kernel.
}
\eqn{G_i(.,.)} can take two forms:
\itemize{
\item For time-independent spatial excitement (\code{time_independent = TRUE}),
\eqn{G_i(s-x_i, t - \tau_i) = f(s - x_i)}
where \eqn{f} is the density function of \eqn{\textrm{N}(0, \Sigma)}.
\item For time-dependent spatial excitement (\code{time_independent = FALSE}),
\eqn{G_i(s-x_i, t - \tau_i) = f(s - x_i)}
where \eqn{f} is the density function of \eqn{\textrm{N}(0, (t-\tau_i)\Sigma)}.
}
}
\examples{
\donttest{
## No GMRF
if(requireNamespace("fmesher")){
data(xyt, package = "stelfi")
N <- 50
locs <- data.frame(x = xyt$x[1:N], y = xyt$y[1:N])
times <- xyt$t[1:N]
domain <- sf::st_as_sf(xyt$window)
bnd <- fmesher::fm_as_segm(as.matrix(sf::st_coordinates(domain)[, 1:2]))
smesh <- fmesher::fm_mesh_2d(boundary = bnd, max.edge = 0.75, cutoff = 0.3) 
param <- list( mu = 3, alpha = 1, beta = 3, xsigma = 0.2, ysigma = 0.2, rho = 0.8)
fit <- fit_stelfi(times = times, locs = locs, sf = domain, smesh = smesh, parameters = param) 
get_coefs(fit)
## GMRF
param <- list( mu = 5, alpha = 1, beta = 3, kappa = 0.9, tau = 1, xsigma = 0.2,
ysigma = 0.2, rho = 0.8)
fit <- fit_stelfi(times = times, locs = locs, sf = domain, smesh = smesh,
parameters = param, GMRF = TRUE)
get_coefs(fit)
}
}
}
\seealso{
\code{\link{fit_hawkes}} and \code{\link{fit_lgcp}}
}
