#' Calculates the `model probabilities of the different GAM models generated by `evaluate_models`
#'
#' @param res_tab a table generated by `evaluate_models`
#' @param n the number of models to retain and generate probabilities for
#'
#' @return A ranked data table in `tibble` format of the top `n` models, their form, BIC and model or (`Pr(M|D)`) or relative (`Pr(M)`) probability value. Model probability indicates the probability of the each model being the correct model and the relative probabilities provide a measure of the doubt about the differences in model specification, when compared to the best or highest ranked model. The relative probabilities are needed when large BIC values generate near zero probability values.
#'
#' @importFrom dplyr rename
#' @importFrom dplyr arrange
#' @importFrom dplyr tibble
#' @importFrom dplyr relocate
#' @importFrom dplyr slice_head
#' @importFrom dplyr across
#'
#' @examples
#' library(dplyr)
#' library(purrr)
#' library(glue)
#' library(mgcv)
#' data(productivity)
#' data = productivity |> filter(year == "1970")
#' svc_res_gam = evaluate_models(data, STVC = FALSE)
#' mod_comp_svc <- gam_model_probs(svc_res_gam, n = 10)
#' # print out the terms
#' mod_comp_svc|> select(-f)
#' @export
gam_model_probs <- function(res_tab, n = 10) {
  bic = NULL
  Prob = NULL
  Rank = NULL
  `Pr(M|D)` = NULL
  nm <- names(res_tab)
  len = length(nm)
  mod_comp <-
    tibble(res_tab) |>
    rename(BIC = bic) |>
    arrange(BIC) |>
    # calculate absolute model probability
    mutate(Prob= exp(-BIC/2)) |>
    mutate(Prob = round(Prob/sum(Prob), 3)) |>
    rename(`Pr(M|D)`=Prob)
  # transpose the indices to to model terms
  # rank and return the top n results
  int_terms <- \(x) c("Fixed","s_S",  "s_T", "s_T + S_S", "s_ST")[x]
  var_terms <- \(x) c("---", "Fixed","s_S", "s_T", "s_T + s_S", "s_ST")[x]
  out_tab <-
    mod_comp |>
    slice_head(n = n) |>
    mutate(across(nm[2]:nm[len-2],var_terms)) |>
    mutate(across(nm[1]:nm[1],int_terms)) |>
    mutate(Rank = 1:n()) |>
    relocate(Rank)
  # if too many models Pr(M|D) is too small
  # then calculate relative probabilities
  # ie relative to the top ranked model
  if (is.na(sum(out_tab$`Pr(M|D)`))) {
    out_tab <- out_tab |> select(-`Pr(M|D)`)
    prob_vec= NULL
    for(i in 2:n) {
      p1 = exp(-(out_tab$BIC[i]-out_tab$BIC[1])/2)
      p1 = p1/(1+p1)
      prob_vec = c(prob_vec, p1)
    }
    out_tab$`Pr(M)` = c("--", paste0(format(round(prob_vec, digits=3), nsmall = 1)))
  }
  out_tab
}

