#' Augment data according to a tidied model
#'
#' Given an R statistical model or other non-tidy object, add columns to the
#' original dataset such as predictions, residuals and cluster assignments.
#'
#' @details
#' `sw_augment()` is a wrapper for `broom::augment()`. The benefit of `sw_augment`
#' is that it has methods for various time-series model classes such as
#' `HoltWinters`, `ets`, `Arima`, etc.
#'
#' For non-time series, `sw_augment()` defaults to `broom::augment()`.
#' The only difference is that the return is a tibble.
#'
#' Note that by convention the first argument is almost always \code{data},
#' which specifies the original data object. This is not part of the S3
#' signature, partly because it prevents \link{rowwise_df_tidiers} from
#' taking a column name as the first argument.
#'
#' @seealso [broom::augment()]
#' @param x model or other R object to convert to data frame
#' @param ... other arguments passed to methods
#' @export
sw_augment <- function(x, ...) UseMethod("sw_augment")


#' Default augment method
#'
#' By default, `sw_augment()` uses [broom::augment()] to convert its output.
#'
#'
#' @param x an object to be tidied
#' @param ... extra arguments passed to `broom::augment()`
#'
#' @return A tibble generated by [broom::augment()]
#'
#' @export
sw_augment.default <- function(x, ...) {
    broom::augment(x, ...) %>%
        tibble::as_tibble()
}
