% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_uid.R
\name{get_uid}
\alias{get_uid}
\alias{as.uid}
\alias{as.uid.uid}
\alias{as.uid.character}
\alias{as.uid.list}
\alias{as.uid.numeric}
\alias{as.uid.data.frame}
\alias{as.data.frame.uid}
\alias{get_uid_}
\title{Get the UID codes from NCBI for taxonomic names.}
\usage{
get_uid(sciname, ask = TRUE, messages = TRUE, rows = NA,
  modifier = NULL, rank_query = NULL, division_filter = NULL,
  rank_filter = NULL, key = NULL, ...)

as.uid(x, check = TRUE)

\method{as.uid}{uid}(x, check = TRUE)

\method{as.uid}{character}(x, check = TRUE)

\method{as.uid}{list}(x, check = TRUE)

\method{as.uid}{numeric}(x, check = TRUE)

\method{as.uid}{data.frame}(x, check = TRUE)

\method{as.data.frame}{uid}(x, ...)

get_uid_(sciname, messages = TRUE, rows = NA, key = NULL, ...)
}
\arguments{
\item{sciname}{character; scientific name. Or, a \code{taxon_state}
object (see \link{taxon-state})}

\item{ask}{logical; should get_uid be run in interactive mode? If TRUE and
more than one TSN is found for the species, the user is asked for input. If
FALSE NA is returned for multiple matches.}

\item{messages}{logical; If \code{TRUE} (default) the actual taxon queried is
printed on the console.}

\item{rows}{numeric; Any number from 1 to infinity. If the default NA, all
rows are considered. Note that this function still only gives back a uid
class object with one to many identifiers. See \code{\link[=get_uid_]{get_uid_()}} to get back
all, or a subset, of the raw data that you are presented during the ask
process.}

\item{modifier}{(character) A modifier to the \code{sciname} given. Options
include: Organism, Scientific Name, Common Name, All Names, Division,
Filter, Lineage, GC, MGC, Name Tokens, Next Level, PGC, Properties, Rank,
Subtree, Synonym, Text Word. These are not checked, so make sure they are
entered correctly, as is.}

\item{rank_query}{(character) A taxonomic rank name to modify the query sent
to NCBI. See \code{\link[=rank_ref]{rank_ref()}} for possible options. Though note that
some data sources use atypical ranks, so inspect the data itself for
options. Optional. See \code{Querying} below.}

\item{division_filter}{(character) A division (aka phylum) name to filter
data after retrieved from NCBI. Optional. See \code{Filtering} below.}

\item{rank_filter}{(character) A taxonomic rank name to filter data after
retrieved from NCBI. See \code{\link[=rank_ref]{rank_ref()}} for possible options.
Though note that some data sources use atypical ranks, so inspect the data
itself for options. Optional. See \code{Filtering} below.}

\item{key}{(character) NCBI Entrez API key. optional. See Details.}

\item{...}{Ignored}

\item{x}{Input to \code{\link[=as.uid]{as.uid()}}}

\item{check}{logical; Check if ID matches any existing on the DB, only used
in \code{\link[=as.uid]{as.uid()}}}
}
\value{
A vector of taxonomic identifiers as an S3 class.
If a taxon is not found an \code{NA} is given. If more than one identifier
is found the function asks for user input if \code{ask = TRUE}, otherwise
returns \code{NA}. If \code{ask=FALSE} and \code{rows} does not equal
\code{NA}, then a data.frame is given back, but not of the uid class, which
you can't pass on to other functions as you normally can.

See \code{\link{get_id_details}} for further details including
attributes and exceptions
}
\description{
Retrieve the Unique Identifier (UID) of a taxon from NCBI taxonomy browser.
}
\section{Querying}{
 The parameter \code{rank_query} is used in the search sent
to NCBI, whereas \code{rank_filter} filters data after it comes back. The
parameter \code{modifier} adds modifiers to the name. For example,
\code{modifier="Organism"} adds that to the name, giving e.g.,
\code{Helianthus[Organism]}.
}

\section{Filtering}{
 The parameters \code{division_filter} and
\code{rank_filter} are not used in the search to the data provider, but are
used in filtering the data down to a subset that is closer to the target
you want. For all these parameters, you can use regex strings since we use
\code{\link[=grep]{grep()}} internally to match. Filtering narrows down to the set
that matches your query, and removes the rest.
}

\section{Beware}{
 NCBI does funny things sometimes. E.g., if you search on
Fringella morel, a slight misspelling of the genus name, and a non-existent
epithet, NCBI gives back a morel fungal species. In addition, NCBI doesn't
really do fuzzy searching very well, so if there is a slight mis-spelling
in your names, you likely won't get what you are expecting. The lesson:
clean your names before using this function. Other data sources are better
about fuzzy matching.
}

\section{Authentication}{

See \link{taxize-authentication} for help on authentication

Note that even though you can't pass in your key to \code{as.uid} functions,
we still use your Entrez API key if you have it saved as an R option
or environment variable.
}

\examples{
\dontrun{
get_uid(c("Chironomus riparius", "Chaetopteryx"))
get_uid(c("Chironomus riparius", "aaa vva"))

# When not found
get_uid("howdy")
get_uid(c("Chironomus riparius", "howdy"))

# Narrow down results to a division or rank, or both
## By modifying the query
### w/ modifiers to the name
get_uid(sciname = "Aratinga acuticauda", modifier = "Organism")
get_uid(sciname = "bear", modifier = "Common Name")

### w/ rank query
get_uid(sciname = "Pinus", rank_query = "genus")
get_uid(sciname = "Pinus", rank_query = "subgenus")
### division query doesn't really work, for unknown reasons, so not available

## By filtering the result
## Echinacea example
### Results w/o narrowing
get_uid("Echinacea")
### w/ division
get_uid(sciname = "Echinacea", division_filter = "eudicots")
get_uid(sciname = "Echinacea", division_filter = "sea urchins")

## Satyrium example
### Results w/o narrowing
get_uid(sciname = "Satyrium")
### w/ division
get_uid(sciname = "Satyrium", division_filter = "monocots")
get_uid(sciname = "Satyrium", division_filter = "butterflies")

## Rank example
get_uid(sciname = "Pinus")
get_uid(sciname = "Pinus", rank_filter = "genus")
get_uid(sciname = "Pinus", rank_filter = "subgenus")

# Fuzzy filter on any filtering fields
## uses grep on the inside
get_uid("Satyrium", division_filter = "m")

# specify rows to limit choices available
get_uid('Dugesia') # user prompt needed
get_uid('Dugesia', rows=1) # 2 choices, so returns only 1 row, so no choices
get_uid('Dugesia', ask = FALSE) # returns NA for multiple matches

# Go to a website with more info on the taxon
res <- get_uid("Chironomus riparius")
browseURL(attr(res, "uri"))

# Convert a uid without class information to a uid class
as.uid(get_uid("Chironomus riparius")) # already a uid, returns the same
as.uid(get_uid(c("Chironomus riparius","Pinus contorta"))) # same
as.uid(315567) # numeric
as.uid(c(315567,3339,9696)) # numeric vector, length > 1
as.uid("315567") # character
as.uid(c("315567","3339","9696")) # character vector, length > 1
as.uid(list("315567","3339","9696")) # list, either numeric or character
## dont check, much faster
as.uid("315567", check=FALSE)
as.uid(315567, check=FALSE)
as.uid(c("315567","3339","9696"), check=FALSE)
as.uid(list("315567","3339","9696"), check=FALSE)

(out <- as.uid(c(315567,3339,9696)))
data.frame(out)
as.uid( data.frame(out) )

# Get all data back
get_uid_("Puma concolor")
get_uid_("Dugesia")
get_uid_("Dugesia", rows=2)
get_uid_("Dugesia", rows=1:2)
get_uid_(c("asdfadfasd","Pinus contorta"))

# use curl options
get_uid("Quercus douglasii", verbose = TRUE)
}
}
\seealso{
\code{\link[=classification]{classification()}}

Other taxonomic-ids: \code{\link{get_boldid}},
  \code{\link{get_colid}}, \code{\link{get_eolid}},
  \code{\link{get_gbifid}}, \code{\link{get_ids}},
  \code{\link{get_iucn}}, \code{\link{get_natservid}},
  \code{\link{get_nbnid}}, \code{\link{get_pow}},
  \code{\link{get_tolid}}, \code{\link{get_tpsid}},
  \code{\link{get_tsn}}, \code{\link{get_wiki}},
  \code{\link{get_wormsid}}
}
\author{
Eduard Szoecs, \email{eduardszoecs@gmail.com}
}
\concept{taxonomic-ids}
