% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/teal_data-get_code.R
\name{get_code,teal_data-method}
\alias{get_code,teal_data-method}
\title{Get code from \code{teal_data} object}
\usage{
\S4method{get_code}{teal_data}(object, deparse = TRUE, datanames = NULL)
}
\arguments{
\item{object}{(\code{teal_data})}

\item{deparse}{(\code{logical}) flag specifying whether to return code as \code{character} (\code{deparse = TRUE}) or as
\code{expression} (\code{deparse = FALSE}).}

\item{datanames}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}} (\code{character}) vector of dataset names to return the code for.
For more details see the "Extracting dataset-specific code" section.}
}
\value{
Either a character string or an expression. If \code{datanames} is used to request a specific dataset,
only code that \emph{creates} that dataset (not code that uses it) is returned. Otherwise, all contents of \verb{@code}.
}
\description{
Retrieve code from \code{teal_data} object.
}
\details{
Retrieve code stored in \verb{@code}, which (in principle) can be used to recreate all objects found in \verb{@env}.
Use \code{datanames} to limit the code to one or more of the datasets enumerated in \verb{@datanames}.
If the code has not passed verification (with \code{\link[=verify]{verify()}}), a warning will be prepended.
}
\section{Extracting dataset-specific code}{

When \code{datanames} is specified, the code returned will be limited  to the lines needed to \emph{create}
the requested datasets. The code stored in the \verb{@code} slot is analyzed statically to determine
which lines the datasets of interest depend upon. The analysis works well when objects are created
with standard infix assignment operators (see \code{?assignOps}) but it can fail in some situations.

Consider the following examples:

\emph{Case 1: Usual assignments.}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- teal_data() |>
  within(\{
    foo <- function(x) \{
      x + 1
    \}
    x <- 0
    y <- foo(x)
  \})
get_code(data, datanames = "y")
}\if{html}{\out{</div>}}

\code{x} has no dependencies, so \code{get_code(data, datanames = "x")} will return only the second call.\cr
\code{y} depends on \code{x} and \code{foo}, so \code{get_code(data, datanames = "y")} will contain all three calls.

\emph{Case 2: Some objects are created by a function's side effects.}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- teal_data() |>
  within(\{
    foo <- function() \{
      x <<- x + 1
    \}
    x <- 0
    foo()
    y <- x
  \})
get_code(data, datanames = "y")
}\if{html}{\out{</div>}}

Here, \code{y} depends on \code{x} but \code{x} is modified by \code{foo} as a side effect (not by reassignment)
and so \code{get_code(data, datanames = "y")} will not return the \code{foo()} call.\cr
To overcome this limitation, code dependencies can be specified manually.
Lines where side effects occur can be flagged by adding "\verb{# @linksto <object name>}" at the end.\cr
Note that \code{within} evaluates code passed to \code{expr} as is and comments are ignored.
In order to include comments in code one must use the \code{eval_code} function instead.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- teal_data() |>
  eval_code("
    foo <- function() \{
      x <<- x + 1
    \}
    x <- 0
    foo() # @linksto x
    y <- x
  ")
get_code(data, datanames = "y")
}\if{html}{\out{</div>}}

Now the \code{foo()} call will be properly included in the code required to recreate \code{y}.

Note that two functions that create objects as side effects, \code{assign} and \code{data}, are handled automatically.

Here are known cases where manual tagging is necessary:
\itemize{
\item non-standard assignment operators, \emph{e.g.} \verb{\%<>\%}
\item objects used as conditions in \code{if} statements: \verb{if (<condition>)}
\item objects used to iterate over in \code{for} loops: \verb{for(i in <sequence>)}
\item creating and evaluating language objects, \emph{e.g.} \verb{eval(<call>)}
}
}

\examples{
tdata1 <- teal_data()
tdata1 <- within(tdata1, {
  a <- 1
  b <- a^5
  c <- list(x = 2)
})
get_code(tdata1)
get_code(tdata1, datanames = "a")
get_code(tdata1, datanames = "b")

tdata2 <- teal_data(x1 = iris, code = "x1 <- iris")
get_code(tdata2)
get_code(verify(tdata2))

}
