% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/teal_data-get_code.R
\name{get_code,teal_data-method}
\alias{get_code,teal_data-method}
\title{Get code from \code{teal_data} object}
\usage{
\S4method{get_code}{teal_data}(
  object,
  deparse = TRUE,
  names = NULL,
  datanames = lifecycle::deprecated(),
  ...
)
}
\arguments{
\item{object}{(\code{teal_data})}

\item{deparse}{(\code{logical}) flag specifying whether to return code as \code{character} (\code{deparse = TRUE}) or as
\code{expression} (\code{deparse = FALSE}).}

\item{names}{(\code{character}) Successor of \code{datanames}. Vector of dataset names to return the code for.
For more details see the "Extracting dataset-specific code" section.}

\item{datanames}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} (\code{character}) vector of dataset names to return the code for.
For more details see the "Extracting dataset-specific code" section. Use \code{names} instead.}

\item{...}{Parameters passed to internal methods. Currently, the only supported parameter is \code{check_names}
(\code{logical(1)}) flag,  which is \code{TRUE} by default. Function warns about missing objects, if they do not exist in
\code{code} but are passed in \code{datanames}. To remove the warning, set \code{check_names = FALSE}.}
}
\value{
Either a character string or an expression. If \code{names} is used to request a specific dataset,
only code that \emph{creates} that dataset (not code that uses it) is returned. Otherwise, all contents of \verb{@code}.
}
\description{
Retrieve code from \code{teal_data} object.
}
\details{
Retrieve code stored in \verb{@code}, which (in principle) can be used to recreate
all objects found in the environment (\verb{@.xData}).
Use \code{names} to limit the code to one or more of the datasets enumerated in
the environment.
}
\section{Extracting dataset-specific code}{

When \code{names} is specified, the code returned will be limited  to the lines needed to \emph{create}
the requested datasets. The code stored in the \verb{@code} slot is analyzed statically to determine
which lines the datasets of interest depend upon. The analysis works well when objects are created
with standard infix assignment operators (see \code{?assignOps}) but it can fail in some situations.

Consider the following examples:

\emph{Case 1: Usual assignments.}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- teal_data() |>
  within(\{
    foo <- function(x) \{
      x + 1
    \}
    x <- 0
    y <- foo(x)
  \})
get_code(data, names = "y")
}\if{html}{\out{</div>}}

\code{x} has no dependencies, so \code{get_code(data, names = "x")} will return only the second call.\cr
\code{y} depends on \code{x} and \code{foo}, so \code{get_code(data, names = "y")} will contain all three calls.

\emph{Case 2: Some objects are created by a function's side effects.}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- teal_data() |>
  within(\{
    foo <- function() \{
      x <<- x + 1
    \}
    x <- 0
    foo()
    y <- x
  \})
get_code(data, names = "y")
}\if{html}{\out{</div>}}

Here, \code{y} depends on \code{x} but \code{x} is modified by \code{foo} as a side effect (not by reassignment)
and so \code{get_code(data, names = "y")} will not return the \code{foo()} call.\cr
To overcome this limitation, code dependencies can be specified manually.
Lines where side effects occur can be flagged by adding "\verb{# @linksto <object name>}" at the end.\cr
Note that \code{within} evaluates code passed to \code{expr} as is and comments are ignored.
In order to include comments in code one must use the \code{eval_code} function instead.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- teal_data() |>
  eval_code("
    foo <- function() \{
      x <<- x + 1
    \}
    x <- 0
    foo() # @linksto x
    y <- x
  ")
get_code(data, names = "y")
}\if{html}{\out{</div>}}

Now the \code{foo()} call will be properly included in the code required to recreate \code{y}.

Note that two functions that create objects as side effects, \code{assign} and \code{data}, are handled automatically.

Here are known cases where manual tagging is necessary:
\itemize{
\item non-standard assignment operators, \emph{e.g.} \verb{\%<>\%}
\item objects used as conditions in \code{if} statements: \verb{if (<condition>)}
\item objects used to iterate over in \code{for} loops: \verb{for(i in <sequence>)}
\item creating and evaluating language objects, \emph{e.g.} \verb{eval(<call>)}
}
}

\examples{
tdata1 <- teal_data()
tdata1 <- within(tdata1, {
  a <- 1
  b <- a^5
  c <- list(x = 2)
})
get_code(tdata1)
get_code(tdata1, names = "a")
get_code(tdata1, names = "b")

tdata2 <- teal_data(x1 = iris, code = "x1 <- iris")
get_code(tdata2)
get_code(verify(tdata2))

}
