#' Filter search result and keep only items with matching property and Q id
#'
#' @param search A data frame generated by `tw_search()`, or a search query. If a data frame is given, language and limits are ignore.
#' @param p A character vector of length 1, a property. Must always start with the capital letter "P", e.g. "P31" for "instance of".
#' @param q A character vector of length 1, a wikidata id. Must always start with the capital letter "Q", e.g. "Q5" for "human being".
#' @param language Language to be used for the search.
#' @param limit Maximum numbers of responses to be given.
#' @param cache Defaults to NULL. If given, it should be given either TRUE or FALSE. Typically set with `tw_enable_cache()` or `tw_disable_cache()`.
#' @param wait In seconds, defaults to 0. Time to wait between queries to Wikidata. If data are cached locally, wait time is not applied. If you are running many queries systematically you may want to add some waiting time between queries.
#'
#' @return A data frame with three columns, `id`, `label`, and `description`, filtered by the above criteria.
#' @export
#'
#' @examples
#' tw_search("Margaret Mead", limit = 3) %>%
#'   tw_filter(p = "P31", q = "Q5")
tw_filter <- function(search,
                      p,
                      q,
                      language = "en",
                      limit = 10,
                      wait = 0,
                      cache = NULL) {
  search_result <- tidywikidatar::tw_check_search(
    search = search,
    language = language,
    limit = limit,
    wait = wait,
    cache = tw_check_cache(cache)
  )

  if (nrow(search_result) == 0) {
    return(search_result)
  }

  filtered_id <- purrr::map_dfr(
    .x = search_result$id,
    .f =
      function(current_id) {
        current_value <- tidywikidatar::tw_get_property(
          id = current_id,
          p = p,
          cache = tw_check_cache(cache),
          wait = wait
        )

        if (is.null(current_value)) {
          tibble::tibble(id = as.character("NA"))
        } else if (length(current_value) == 0) {
          tibble::tibble(id = as.character("NA"))
        } else if (is.element(q, current_value) == TRUE) {
          tibble::tibble(id = current_id)
        } else {
          tibble::tibble(id = as.character("NA"))
        }
      }
  ) %>%
    tidyr::drop_na()

  search_result %>%
    dplyr::semi_join(
      y = filtered_id,
      by = "id"
    )
}

#' Filter search result and keep only and keep only the first match
#'
#' Same as `tw_filter()`, but consistently returns data frames with a single row.
#'
#' @param search A data frame generated by `tw_search()`, or a search query. If a data frame is given, language and limits are ignore.
#' @param p A character vector of length 1, a property. Must always start with the capital letter "P", e.g. "P31" for "instance of".
#' @param q A character vector of length 1, a wikidata id. Must always start with the capital letter "Q", e.g. "Q5" for "human being".
#' @param language Language to be used for the search.
#' @param limit Maximum numbers of responses to be given.
#' @param cache Defaults to NULL. If given, it should be given either TRUE or FALSE. Typically set with `tw_enable_cache()` or `tw_disable_cache()`
#' @param wait In seconds, defaults to 0. Time to wait between queries to Wikidata. If data are cached locally, wait time is not applied. If you are running many queries systematically you may want to add some waiting time between queries.
#'
#' @return A data frame with one row and three columns, `id`, `label`, and `description`, filtered by the above criteria.
#' @export
#'
#' @examples
#' tw_search("Margaret Mead") %>%
#'   tw_filter_first(p = "P31", q = "Q5")
tw_filter_first <- function(search,
                            p,
                            q,
                            language = "en",
                            limit = 10,
                            wait = 0,
                            cache = NULL) {
  search_result <- tidywikidatar::tw_check_search(
    search = search,
    language = language,
    limit = limit,
    wait = wait,
    cache = tw_check_cache(cache)
  )

  if (nrow(search_result) == 0) {
    return(search_result)
  }

  first_match_id <- purrr::detect(
    .x = seq_along(search_result$id),
    .f = function(current_row_number) {
      search_result %>%
        dplyr::slice(current_row_number) %>%
        tidywikidatar::tw_filter(p = p, q = q) %>%
        nrow() %>%
        `>`(0)
    }
  )

  if (is.null(first_match_id)) {
    search_result %>%
      dplyr::slice(0)
  } else {
    search_result %>%
      dplyr::slice(first_match_id)
  }
}

#' Filter search result and keep only people
#'
#' A wrapper of `tw_filter()` that defaults to keep only "instance of" (P31) "human being" (Q5).
#'
#' @param search A data frame generated by `tw_search()`, or a search query. If a data frame is given, language and limits are ignore.
#' @param language Language to be used for the search.
#' @param stop_at_first Logical, defaults to TRUE. If TRUE, returns only the first match from the search that satisfies the criteria.
#' @param limit Maximum numbers of responses to be given.
#' @param wait In seconds, defaults to 0. Time to wait between queries to Wikidata. If data are cached locally, wait time is not applied. If you are running many queries systematically you may want to add some waiting time between queries.
#'
#' @return A data frame with three columns, `id`, `label`, and `description`; all rows refer to a human being.
#' @export
#'
#' @examples
#' tw_search("Ruth Benedict")
#'
#' tw_search("Ruth Benedict") %>%
#'   tw_filter_people()
tw_filter_people <- function(search,
                             language = "en",
                             limit = 10,
                             stop_at_first = TRUE,
                             wait = 0) {
  if (stop_at_first == TRUE) {
    tidywikidatar::tw_filter_first(
      search = search,
      p = "P31",
      q = "Q5",
      language = language,
      limit = limit,
      wait = wait
    )
  } else {
    tidywikidatar::tw_filter(
      search = search,
      p = "P31",
      q = "Q5",
      language = language,
      limit = limit,
      wait = wait
    )
  }
}
