% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dplyr-pad_by_time.R
\name{pad_by_time}
\alias{pad_by_time}
\title{Insert time series rows with regularly spaced timestamps}
\usage{
pad_by_time(
  .data,
  .date_var,
  .by = "auto",
  .pad_value = NA,
  .fill_na_direction = c("none", "down", "up", "downup", "updown"),
  .start_date = NULL,
  .end_date = NULL
)
}
\arguments{
\item{.data}{A tibble with a time-based column.}

\item{.date_var}{A column containing date or date-time values to pad}

\item{.by}{Either "auto", a time-based frequency like "year", "month", "day", "hour", etc,
or a time expression like "5 min", or "7 days". See Details.}

\item{.pad_value}{Fills in padded values. Default is \code{NA}.}

\item{.fill_na_direction}{Users can provide an \code{NA} fill strategy using \code{tidyr::fill()}.
Possible values: \code{'none'}, \code{'down'}, \code{'up'}, \code{'downup'}, \code{'updown'}. Default: \code{'none'}}

\item{.start_date}{Specifies the start of the padded series.
If NULL it will use the lowest value of the input variable.}

\item{.end_date}{Specifies the end of the padded series.
If NULL it will use the highest value of the input variable.}
}
\value{
A \code{tibble} or \code{data.frame} with rows containing missing timestamps added.
}
\description{
The easiest way to fill in missing timestamps or convert to a more
granular period (e.g. quarter to month). Wraps the \code{padr::pad()} function
for padding tibbles.
}
\details{
\strong{Padding Missing Observations}

The most common use case for \code{pad_by_time()} is to add rows where timestamps
are missing. This could be from sales data that have missing values on weekends and holidays.
Or it could be high frequency data where observations are irregularly spaced and should be
reset to a regular frequency.

\strong{Going from Low to High Frequency}

The second use case is going from a low frequency (e.g. day) to high frequency (e.g. hour).
This is possible by supplying a higher frequency to \code{pad_by_time()}.

\strong{Interval, .by}

Padding can be applied in the following ways:
\itemize{
\item \code{.by = "auto"} - \code{pad_by_time()} will detect the time-stamp frequency and apply padding.
\item The eight intervals in are: year, quarter, month, week, day, hour, min, and sec.
\item Intervals like 5 minutes, 6 hours, 10 days are possible.
}

\strong{Pad Value, .pad_value}

A pad value can be supplied that fills in missing numeric data. Note that this is only applied to numeric columns.

\strong{Fill NA Direction, .fill_na_directions}

Uses \code{tidyr::fill()} to fill missing observations using a fill strategy.
}
\examples{
library(dplyr)

# Create a quarterly series with 1 missing value
missing_data_tbl <- tibble::tibble(
    date = tk_make_timeseries("2014-01-01", "2015-01-01", by = "quarter"),
    value = 1:5
) \%>\%
    slice(-4) # Lose the 4th quarter on purpose
missing_data_tbl


# Detects missing quarter, and pads the missing regularly spaced quarter with NA
missing_data_tbl \%>\% pad_by_time(date, .by = "quarter")

# Can specify a shorter period. This fills monthly.
missing_data_tbl \%>\% pad_by_time(date, .by = "month")

# Can let pad_by_time() auto-detect date and period
missing_data_tbl \%>\% pad_by_time()

# Can specify a .pad_value
missing_data_tbl \%>\% pad_by_time(date, .by = "quarter", .pad_value = 0)

# Can then impute missing values
missing_data_tbl \%>\%
    pad_by_time(date, .by = "quarter") \%>\%
    mutate(value = ts_impute_vec(value, period = 1))

# Can specify a custom .start_date and .end_date
missing_data_tbl \%>\%
   pad_by_time(date, .by = "quarter", .start_date = "2013", .end_date = "2015-07-01")

# Can specify a tidyr::fill() direction
missing_data_tbl \%>\%
   pad_by_time(date, .by = "quarter",
               .fill_na_direction = "downup",
               .start_date = "2013", .end_date = "2015-07-01")

# --- GROUPS ----

# Apply standard NA padding to groups
FANG \%>\%
    group_by(symbol) \%>\%
    pad_by_time(.by = "day")

# Apply constant pad value
FANG \%>\%
    group_by(symbol) \%>\%
    pad_by_time(.by = "day", .pad_value = 0)

# Apply filled padding to groups
FANG \%>\%
    group_by(symbol) \%>\%
    pad_by_time(.by = "day", .fill_na_direction = "down")

}
\references{
\itemize{
\item This function wraps the \code{padr::pad()} function developed by Edwin Thoen.
}
}
\seealso{
Imputation:
\itemize{
\item \code{\link[=ts_impute_vec]{ts_impute_vec()}} - Impute missing values for time series.
}

Time-Based dplyr functions:
\itemize{
\item \code{\link[=summarise_by_time]{summarise_by_time()}} - Easily summarise using a date column.
\item \code{\link[=mutate_by_time]{mutate_by_time()}} - Simplifies applying mutations by time windows.
\item \code{\link[=pad_by_time]{pad_by_time()}} - Insert time series rows with regularly spaced timestamps
\item \code{\link[=filter_by_time]{filter_by_time()}} - Quickly filter using date ranges.
\item \code{\link[=filter_period]{filter_period()}} - Apply filtering expressions inside periods (windows)
\item \code{\link[=slice_period]{slice_period()}} - Apply slice inside periods (windows)
\item \code{\link[=condense_period]{condense_period()}} - Convert to a different periodicity
\item \code{\link[=between_time]{between_time()}} - Range detection for date or date-time sequences.
\item \code{\link[=slidify]{slidify()}} - Turn any function into a sliding (rolling) function
}
}
