% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nn-conv.R
\name{nn_conv_transpose1d}
\alias{nn_conv_transpose1d}
\title{ConvTranspose1D}
\usage{
nn_conv_transpose1d(
  in_channels,
  out_channels,
  kernel_size,
  stride = 1,
  padding = 0,
  output_padding = 0,
  groups = 1,
  bias = TRUE,
  dilation = 1,
  padding_mode = "zeros"
)
}
\arguments{
\item{in_channels}{(int): Number of channels in the input image}

\item{out_channels}{(int): Number of channels produced by the convolution}

\item{kernel_size}{(int or tuple): Size of the convolving kernel}

\item{stride}{(int or tuple, optional): Stride of the convolution. Default: 1}

\item{padding}{(int or tuple, optional): \code{dilation * (kernel_size - 1) - padding} zero-padding
will be added to both sides of the input. Default: 0}

\item{output_padding}{(int or tuple, optional): Additional size added to one side
of the output shape. Default: 0}

\item{groups}{(int, optional): Number of blocked connections from input channels to output channels. Default: 1}

\item{bias}{(bool, optional): If \code{True}, adds a learnable bias to the output. Default: \code{TRUE}}

\item{dilation}{(int or tuple, optional): Spacing between kernel elements. Default: 1}

\item{padding_mode}{(string, optional): \code{'zeros'}, \code{'reflect'},
\code{'replicate'} or \code{'circular'}. Default: \code{'zeros'}}
}
\description{
Applies a 1D transposed convolution operator over an input image
composed of several input planes.
}
\details{
This module can be seen as the gradient of Conv1d with respect to its input.
It is also known as a fractionally-strided convolution or
a deconvolution (although it is not an actual deconvolution operation).
\itemize{
\item \code{stride} controls the stride for the cross-correlation.
\item \code{padding} controls the amount of implicit zero-paddings on both
sides for \code{dilation * (kernel_size - 1) - padding} number of points. See note
below for details.
\item \code{output_padding} controls the additional size added to one side
of the output shape. See note below for details.
\item \code{dilation} controls the spacing between the kernel points; also known as the
à trous algorithm. It is harder to describe, but this \href{https://github.com/vdumoulin/conv_arithmetic}{link}
has a nice visualization of what \code{dilation} does.
\item \code{groups} controls the connections between inputs and outputs.
\code{in_channels} and \code{out_channels} must both be divisible by
\code{groups}. For example,
\itemize{
\item At groups=1, all inputs are convolved to all outputs.
\item At groups=2, the operation becomes equivalent to having two conv
layers side by side, each seeing half the input channels,
and producing half the output channels, and both subsequently
concatenated.
\item At groups= \code{in_channels}, each input channel is convolved with
its own set of filters (of size
\eqn{\left\lfloor\frac{out\_channels}{in\_channels}\right\rfloor}).
}
}
}
\note{
Depending of the size of your kernel, several (of the last)
columns of the input might be lost, because it is a valid \code{cross-correlation}\emph{,
and not a full \code{cross-correlation}}.
It is up to the user to add proper padding.

The \code{padding} argument effectively adds \code{dilation * (kernel_size - 1) - padding}
amount of zero padding to both sizes of the input. This is set so that
when a \code{~torch.nn.Conv1d} and a \code{~torch.nn.ConvTranspose1d}
are initialized with same parameters, they are inverses of each other in
regard to the input and output shapes. However, when \code{stride > 1},
\code{~torch.nn.Conv1d} maps multiple input shapes to the same output
shape. \code{output_padding} is provided to resolve this ambiguity by
effectively increasing the calculated output shape on one side. Note
that \code{output_padding} is only used to find output shape, but does
not actually add zero-padding to output.

In some circumstances when using the CUDA backend with CuDNN, this operator
may select a nondeterministic algorithm to increase performance. If this is
undesirable, you can try to make the operation deterministic (potentially at
a performance cost) by setting \code{torch.backends.cudnn.deterministic = TRUE}.
}
\section{Shape}{

\itemize{
\item Input: \eqn{(N, C_{in}, L_{in})}
\item Output: \eqn{(N, C_{out}, L_{out})} where
\deqn{
  L_{out} = (L_{in} - 1) \times \mbox{stride} - 2 \times \mbox{padding} + \mbox{dilation}
\times (\mbox{kernel\_size} - 1) + \mbox{output\_padding} + 1
}
}
}

\section{Attributes}{

\itemize{
\item weight (Tensor): the learnable weights of the module of shape
\eqn{(\mbox{in\_channels}, \frac{\mbox{out\_channels}}{\mbox{groups}},}
\eqn{\mbox{kernel\_size})}.
The values of these weights are sampled from
\eqn{\mathcal{U}(-\sqrt{k}, \sqrt{k})} where
\eqn{k = \frac{groups}{C_{\mbox{out}} * \mbox{kernel\_size}}}
\item bias (Tensor):   the learnable bias of the module of shape (out_channels).
If \code{bias} is \code{TRUE}, then the values of these weights are
sampled from \eqn{\mathcal{U}(-\sqrt{k}, \sqrt{k})} where
\eqn{k = \frac{groups}{C_{\mbox{out}} * \mbox{kernel\_size}}}
}
}

\examples{
if (torch_is_installed()) {
m <- nn_conv_transpose1d(32, 16, 2)
input <- torch_randn(10, 32, 2)
output <- m(input)
}
}
