% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{transreg}
\alias{transreg}
\title{Penalised regression with multiple sets of prior effects}
\usage{
transreg(
  y,
  X,
  prior,
  family = "gaussian",
  alpha = 1,
  foldid = NULL,
  nfolds = 10,
  scale = "iso",
  stack = "sim",
  sign = FALSE,
  switch = FALSE,
  select = TRUE,
  track = FALSE,
  parallel = FALSE
)
}
\arguments{
\item{y}{target: vector of length \eqn{n} (see \code{family})}

\item{X}{features: matrix with \eqn{n} rows (samples)
and \eqn{p} columns (features)}

\item{prior}{prior coefficients: matrix with \eqn{p} rows (features)
and \eqn{k} columns (sources of co-data)}

\item{family}{character "gaussian" (\eqn{y}: real numbers),
"binomial" (\eqn{y}: 0s and 1s),
or "poisson" (\eqn{y}: non-negative integers);}

\item{alpha}{elastic net mixing parameter (0=ridge, 1=lasso):
number between 0 and 1}

\item{foldid}{fold identifiers: vector of length \eqn{n}
with entries from 1 to \code{nfolds}}

\item{nfolds}{number of folds: positive integer}

\item{scale}{character
"exp" for exponential calibration or
"iso" for isotonic calibration}

\item{stack}{character "sta" (standard stacking) or "sim" (simultaneous stacking)}

\item{sign}{sign discovery procedure: logical
(experimental argument)}

\item{switch}{choose between positive and negative weights for each source: logical}

\item{select}{select from sources: logical}

\item{track}{show intermediate output (messages and plots): logical}

\item{parallel}{logical (see cv.glmnet)}
}
\value{
Returns an object of class \code{transreg}.
Rather than accessing its slots (see list below),
it is recommended to use methods like
\code{\link[=coef.transreg]{coef.transreg()}} and \code{\link[=predict.transreg]{predict.transreg()}}.
\itemize{
\item slot \code{base}:
Object of class \code{glmnet}.
Regression of outcome on features (without prior effects),
with \eqn{1 + p} estimated coefficients
(intercept + features).
\item slot \code{meta.sta}:
\code{NULL} or object of class \code{glmnet}.
Regression of outcome on cross-validated linear predictors
from prior effects and estimated effects,
with \eqn{1 + k + 2} estimated coefficients
(intercept + sources of co-data + lambda_min and lambda_1se).
\item slot \code{meta.sim}:
\code{NULL} or object of class \code{glmnet}.
Regression of outcome on meta-features
(cross-validated linear predictors from prior effects)
and original features,
with \eqn{1 + k + p} estimated coefficients
(intercept + sources of co-data + features).
\item slot \code{prior.calib}:
Calibrated prior effects.
Matrix with \eqn{p} rows and \eqn{k} columns.
\item slot \code{data}:
Original data.
List with slots \code{y}, \code{X} and \code{prior} (see arguments).
\item slot \code{info}:
Information on call.
Data frame with entries
\eqn{n}, \eqn{p}, \eqn{k}, \code{family}, \code{alpha}, \code{scale} and \code{stack}
(see details and arguments).
}
}
\description{
Implements penalised regression with multiple sets of prior effects
}
\details{
\itemize{
\item \eqn{n}: sample size
\item \eqn{p}: number of features
\item \eqn{k}: number of sources
}
}
\examples{
#--- simulation ---
n <- 100; p <- 500
X <- matrix(rnorm(n=n*p),nrow=n,ncol=p)
beta <- rnorm(p)*rbinom(n=p,size=1,prob=0.2)
prior1 <- beta + rnorm(p)
prior2 <- beta + rnorm(p)
y_lin <- X \%*\% beta
y_log <- 1*(y_lin > 0)

#--- single vs multiple priors ---
one <- transreg(y=y_lin,X=X,prior=prior1)
two <- transreg(y=y_lin,X=X,prior=cbind(prior1,prior2))
weights(one)
weights(two)

\donttest{
#--- linear vs logistic regression ---
lin <- transreg(y=y_lin,X=X,prior=prior1,family="gaussian")
log <- transreg(y=y_log,X=X,prior=prior1,family="binomial")
hist(predict(lin,newx=X)) # predicted values
hist(predict(log,newx=X)) # predicted probabilities

#--- ridge vs lasso penalisation ---
ridge <- transreg(y=y_lin,X=X,prior=prior1,alpha=0)
lasso <- transreg(y=y_lin,X=X,prior=prior1,alpha=1)
# initial coefficients (without prior)
plot(x=coef(ridge$base)[-1]) # dense
plot(x=coef(lasso$base)[-1]) # sparse
# final coefficients (with prior)
plot(x=coef(ridge)$beta) # dense
plot(x=coef(lasso)$beta) # not sparse

#--- exponential vs isotonic calibration ---
exp <- transreg(y=y_lin,X=X,prior=prior1,scale="exp")
iso <- transreg(y=y_lin,X=X,prior=prior1,scale="iso")
plot(x=prior1,y=exp$prior.calib)
plot(x=prior1,y=iso$prior.calib)

#--- standard vs simultaneous stacking ---
prior <- c(prior1[1:250],rep(0,250))
sta <- transreg(y=y_lin,X=X,prior=prior,stack="sta")
sim <- transreg(y=y_lin,X=X,prior=prior,stack="sim")
plot(x=coef(sta$base)[-1],y=coef(sta)$beta)
plot(x=coef(sim$base)[-1],y=coef(sim)$beta)}

}
\references{
Armin Rauschenberger
\href{https://orcid.org/0000-0001-6498-4801}{\figure{orcid.png}{height=10px}},
Zied Landoulsi
\href{https://orcid.org/0000-0002-2327-3904}{\figure{orcid.png}{height=10px}},
Mark A. van de Wiel
\href{https://orcid.org/0000-0003-4780-8472}{\figure{orcid.png}{height=10px}},
and Enrico Glaab
\href{https://orcid.org/0000-0003-3977-7469}{\figure{orcid.png}{height=10px}}
(2022).
"Penalised regression with multiple sets of prior effects".
\emph{Bioinformatics (In press).}
\doi{10.48550/arXiv.2212.08581}
\email{armin.rauschenberger@uni.lu}
}
\seealso{
Methods for objects of class \link{transreg}
include \code{\link[=coef.transreg]{coef}}
and \code{\link[=predict.transreg]{predict}}.
}
