% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tvLM.R
\name{tvLM}
\alias{tvLM}
\alias{tvlm-class}
\alias{tvlm}
\title{Time-Varying Coefficients Linear Models}
\usage{
tvLM(formula, z = NULL, ez = NULL, data, bw = NULL, est = c("lc",
  "ll"), tkernel = c("Epa", "Gaussian"), singular.ok = TRUE)
}
\arguments{
\item{formula}{An object of class formula.}

\item{z}{A vector with the smoothing variable.}

\item{ez}{(optional) A scalar or vector with the smoothing estimation values. If 
values are included then the vector \code{z} is used.}

\item{data}{An optional data frame or matrix.}

\item{bw}{An opcional scalar. It represents the bandwidth in
the estimation of trend coefficients. If NULL, it is selected by cross validation.}

\item{est}{The nonparametric estimation method, one of "lc" (default) for linear constant
or "ll" for local linear.}

\item{tkernel}{The type of kernel used in the coefficients estimation method,
one of Epanesnikov ("Epa") or "Gaussian".}

\item{singular.ok}{Logical. If FALSE, a singular model is an error.}
}
\value{
An object of class \code{tvlm}
The object of class \code{tvlm} have the following components:
\item{tvcoef}{A matrix of dimensions}
\item{fitted}{The fitted values.}
\item{residuals}{Estimation residuals.}
\item{x}{A matrix with the regressors data.}
\item{y}{A vector with the dependent variable data.}
\item{z}{A vector with the smoothing variable.}
\item{ez}{A vector with the smoothing estimation variable.}
\item{bw}{Bandwidth of mean estimation.}
\item{est}{Nonparametric estimation methodology.}
\item{tkernel}{Kernel used in estimation.}
\item{level}{Confidence interval range.}
\item{runs}{Number of bootstrap replications.}
\item{tboot}{Type of bootstrap.}
\item{BOOT}{List with all bootstrap replications of \code{tvcoef}, if done.}
}
\description{
\code{tvLM} is used to fit a time-varying coefficients linear model
}
\details{
Models for \code{tvLM} are specified symbolically using the same formula
format than function \code{lm}. A typical model has the form \emph{response} ~ \emph{terms}
where response is the (numeric) response vector and terms is a series of terms which
specifies a linear predictor for response. A terms specification of the form
first + second indicates all the terms in first together with all the terms
in second with duplicates removed. A specification of the form first:second indicates
the set of terms obtained by taking the interactions of all terms in first with all
terms in second. The specification first*second indicates the cross of first and second.
This is the same as first + second + first:second.

A formula has an implied intercept term. To remove this use either
y ~ x - 1 or y ~ 0 + x.
}
\examples{
## Simulate a linear process with time-varying coefficient
## as functions of scaled time.
set.seed(42)
tau <- seq(1:200)/200
beta <- data.frame(beta1 = sin(2*pi*tau), beta2= 2*tau)
X1 <- rnorm(200)
X2 <- rchisq(200, df = 4)
error <- rt(200, df = 10)
y <- apply(cbind(X1, X2)*beta, 1, sum) + error
data <- data.frame(y = y, X1 = X1, X2 = X2)
## Estimate coefficients with lm and tvLM for comparison

coef.lm <- stats::lm(y ~ 0 + X1 + X2, data = data)$coef
tvlm.fit <- tvLM(y ~ 0 + X1 + X2, data = data, bw = 0.29)

## Estimate coefficients of different realized variance models
data("RV")
##Bollerslev et al. (2016) HARQ model
HARQ <- lm (RV ~ RV_lag + I(RV_lag * RQ_lag_sqrt) + RV_week + RV_month, 
                 data = RV)

#Casas et al. (2018) TV-HARQ model
RQ <- RV$RQ_lag_sqrt
tvHARQ <- tvLM (RV ~ RV_lag + RV_week + RV_month, z = RQ, data = RV, 
                     bw = 0.004)
boxplot(data.frame(tvHARQ = tvHARQ$tvcoef[,2] * RV$RV_lag,
                   HARQ = (HARQ$coef[2] + HARQ$coef[3] * RV$RQ_lag_sqrt)*RV$RV_lag),
                   main = expression (RV[t-1]), outline = FALSE)
}
\references{
Bollerslev, T., Patton, A. J. and Quaedvlieg, R. (2016) Exploiting the 
errors: A simple approach for improved volatility forecasting. 
\emph{Journal of Econometrics}, 192, 1-18.

Casas, I., Mao, X. and Veiga, H. (2018) Reexamining financial and economic 
predictability with new estimators of realized variance and variance 
risk premium. Url= http://pure.au.dk/portal/files/123066669/rp18_10.pdf
}
\seealso{
\code{\link{bw}}, \code{\link{tvAR}}, \code{\link{confint}}, 
\code{\link{plot}}, \code{\link{print}} and \code{\link{summary}}
}
\keyword{coefficients}
\keyword{linear}
\keyword{models,}
\keyword{nonparametric}
\keyword{regression}
\keyword{regression,}
\keyword{statistics}
\keyword{time}
\keyword{time-varying}
\keyword{varying}
