% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateGSMAR.R
\name{simulateGSMAR}
\alias{simulateGSMAR}
\title{Simulate values from GMAR, StMAR or G-StMAR process}
\usage{
simulateGSMAR(gsmar, nsimu, initvalues, ntimes = 1)
}
\arguments{
\item{gsmar}{object of class \code{'gsmar'}, generated by function \code{fitGSMAR()} or \code{GSMAR()}.}

\item{nsimu}{a positive integer specifying how many values will be simulated.}

\item{initvalues}{a numeric vector with length \code{>=p} specifying the initial values for the simulation. The \strong{last} element will be used as the
initial value for the first lag, the second last element will be initial value for the second lag, etc. If not specified, initial
values will be simulated from the process's stationary distribution.}

\item{ntimes}{a positive integer specifying how many sets of simulations should be performed. If larger than one then
only the samples are returned. Uses the same initial values for each set. Default is one.}
}
\value{
Returns a list with...
  \describe{
    \item{\code{$sample}}{a numeric vector containing the simulated values.}
    \item{\code{$component}}{a numeric vector the containing the information from which component each value is generated from.}
    \item{\code{$mixingWeights}}{a size (\code{nsimu}\eqn{xM}) matrix containing the mixing weights corresponding to the
     sample so that \eqn{i}:th column is for \eqn{i}:th component.}
  }
  Or if \strong{\code{ntimes>1}} returns a matrix containing the samples so that the \code{i}:th
  sample can be obtained at \code{i}:th column.
}
\description{
\code{simulateGSMAR} simulates values from the specified GMAR, StMAR or G-StMAR process.
}
\examples{
 \donttest{
 # GMAR model
 params13 <- c(1.4, 0.88, 0.26, 2.46, 0.82, 0.74, 5.0, 0.68, 5.2, 0.72, 0.2)
 gmar13 <- GSMAR(p=1, M=3, params=params13, model="GMAR")
 sim13 <- simulateGSMAR(gmar13, nsimu=500)
 ts.plot(sim13$sample)
 ts.plot(sim13$component)
 ts.plot(sim13$mixingWeights, col=rainbow(3), lty=2)

 # Restricted GMAR model
 params12r <- c(1.4, 1.8, 0.88, 0.29, 3.18, 0.84)
 gmar12r <- GSMAR(p=1, M=2, params=params12r, model="GMAR",
 restricted=TRUE)
 sim12r <- simulateGSMAR(gmar12r, nsimu=500)
 ts.plot(sim12r$sample)
 ts.plot(sim12r$component)
 ts.plot(sim12r$mixingWeights, col=rainbow(3), lty=2)

 # G-StMAR model, with initial values
 params12gs <- c(1.38, 0.88, 0.27, 3.8, 0.74, 3.15, 0.8, 3.6)
 gstmar12 <- GSMAR(p=1, M=c(1, 1), params=params12gs,
 model="G-StMAR")
 sim12gs <- simulateGSMAR(gstmar12, nsimu=500, initvalues=5:6)
 ts.plot(sim12gs$sample)
 ts.plot(sim12gs$component)
 ts.plot(sim12gs$mixingWeights, col=rainbow(3), lty=2)

 # Such StMAR(3,2) that the AR coefficients are restricted to be
 # the same for both regimes and that the second AR coefficients are
 # constrained to zero.
 params32trc <- c(2.2, 1.8, 0.88, -0.03, 2.4, 0.27, 0.40, 3.9, 1000)
 stmar32rc <- GSMAR(data=VIX, p=3, M=2, params=params32trc, model="StMAR",
  restricted=TRUE, constraints=matrix(c(1, 0, 0, 0, 0, 1), ncol=2))
 sim32trc <- simulateGSMAR(stmar32rc, nsimu=500)
 ts.plot(sim32trc$sample)
 ts.plot(sim32trc$component)
 ts.plot(sim32trc$mixingWeights, col=rainbow(3), lty=2)

 # Mixture version of Heterogenuous Autoregressive (HAR) model (without data)
 paramsHAR <- c(1, 0.1, 0.2, 0.3, 1, 1, 0.15, 0.25, 0.35, 1, 0.55)
 r1 = c(1, rep(0, 21)); r2 = c(rep(0.2, 5), rep(0, 17)); r3 = rep(1/22, 22)
 R0 = cbind(r1, r2, r3)
 mixhar <- GSMAR(p=22, M=2, params=paramsHAR, model="GMAR", constraints=list(R0, R0))
 simhar <- simulateGSMAR(mixhar, nsimu=1000)
 ts.plot(simhar$sample)
 ts.plot(simhar$component)
 ts.plot(simhar$mixingWeights, col=rainbow(3), lty=2)
}
}
\references{
\itemize{
   \item Galbraith, R., Galbraith, J. 1974. On the inverses of some patterned matrices arising
           in the theory of stationary time series. \emph{Journal of Applied Probability} \strong{11}, 63-71.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2015. Gaussian Mixture Autoregressive model for univariate time series.
           \emph{Journal of Time Series Analysis}, \strong{36}, 247-266.
   \item Lutkepohl H. 2005. New Introduction to Multiple Time Series Analysis.
           \emph{Springer}.
   \item Meitz M., Preve D., Saikkonen P. 2018. A mixture autoregressive model based on Student's t-distribution.
           arXiv:1805.04010 \strong{[econ.EM]}.
   \item There are currently no published references for G-StMAR model, but it's a straightforward generalization with
           theoretical properties similar to GMAR and StMAR models.
 }
}
