"SVAR" <- function(x, estmethod = c("scoring", "logLik"), Amat = NULL, Bmat = NULL, Ra = NULL, Rb = NULL, ra = NULL, rb = NULL, start = NULL, max.iter = 100, conv.crit = 0.1e-6, maxls = 1.0, lrtest = TRUE, ...){
  ## testing of valid arguments
  if(!class(x)=="varest"){
    stop("\nPlease, provide an object of class 'varest',\n generated by function 'VAR()' as input for 'x'.\n")}
  estmethod <- match.arg(estmethod)
  call <- match.call()
  if(estmethod == "scoring"){
    svar <- .SVAR2(x, Ra = Ra, Rb = Rb, ra = ra, rb = ra, start = start, max.iter = max.iter, conv.crit = conv.crit, maxls = maxls, lrtest = lrtest)
  } else {
    svar <- .SVAR1(x, Amat = Amat, Bmat = Bmat, start = start, lrtest = lrtest, ...)
  }
  svar$call <- call
  return(svar)
}

".SVAR1" <-
function(x = x, Amat = NULL, Bmat = NULL, start = NULL, lrtest = TRUE, ...){
  if((is.null(Amat)) && (is.null(Bmat))){
    stop("\nAt least one matrix, either 'Amat' or 'Bmat', must be non-null.\n")
  }
  if ((is.null(Amat)) && !(is.null(Bmat))) {
    Amat <- diag(x$K)
    svartype <- "B-model"
  } else if ((is.null(Bmat)) && !(is.null(Amat))) {
    Bmat <- diag(x$K)
    svartype <- "A-model"
  } else {
    svartype <- "AB-model"
  }
  if(!any(is.na(cbind(Amat, Bmat)))){
    stop("\nNo parameters provided for optimisation, i.e.\nneither 'Amat' nor 'Bmat' does contain na-elements.\n")    
  }
  param.Aidx <- which(is.na(Amat), arr.ind=TRUE)
  param.Bidx <- which(is.na(Bmat), arr.ind=TRUE)
  ifelse(!is.null(nrow(param.Aidx)), params.A <- nrow(param.Aidx), params.A <- 0)
  ifelse(!is.null(nrow(param.Bidx)), params.B <- nrow(param.Bidx), params.B <- 0)
  params <- params.A + params.B
  K <- x$K
  obs <- x$obs
  df <- summary(x$varresult[[1]])$df[2]
  sigma <- crossprod(resid(x)) / df
  if((svartype == "B-model") || (svartype == "A-model")){
    if(K^2 - params <= K*(K-1)/2){
      stop("\nModel is not identified,\nchoose different settings for 'Amat' and/or 'Bmat'.\n")      
    }
  } else if(svartype == "AB-model"){
    if(2*K^2 - params.A + params.B <= K^2 + K*(K-1)/2){
      stop("\nModel is not identified,\nchoose different settings for 'Amat' and/or 'Bmat'.\n")      
    }
  }
  if(is.null(start)) start <- rep(0.1, params)
  start <- as.vector(start)
  if(!(length(start)==params)){
    stop("\nWrong count of starting values provided in 'start'.\nLength of 'start' must be equal to the count of 'na' in 'Amat' and 'Bmat'.\n")
  }
  logLc <- function(coef){
    if(svartype == "B-model"){
      Bmat[param.Bidx] <- coef
    } else if(svartype == "A-model"){
      Amat[param.Aidx] <- coef
    } else if(svartype == "AB-model"){
      if(length(param.Aidx) > 0){
        Amat[param.Aidx] <- coef[c(1:nrow(param.Aidx))]
        if(length(param.Bidx) > 0){
          Bmat[param.Bidx] <- coef[-c(1:nrow(param.Aidx))]
        }
      } else if(length(param.Aidx) == 0){
        Bmat[param.Bidx] <- coef
      }
    }
    const <- -1*(K*obs/2)*log(2*pi)
    logLc <- const + obs/2*log(det(Amat)^2) - obs/2*log(det(Bmat)^2) - obs/2*sum(diag(t(Amat) %*% solve(t(Bmat)) %*% solve(Bmat) %*% Amat %*% sigma))
    return(-logLc)
  }
  opt <- optim(start, logLc, ...)
  Asigma <- matrix(0, nrow = K, ncol = K)
  Bsigma <- matrix(0, nrow = K, ncol = K)
  if(!(is.null(opt$hessian))){
    Sigma <- sqrt(diag(solve(opt$hessian))) 
  }
  if(svartype == "B-model"){
    Bmat[param.Bidx] <- opt$par
    if(!(is.null(opt$hessian))){
      Bsigma[param.Bidx] <- Sigma
    }
  }else if(svartype == "A-model"){
    Amat[param.Aidx] <- opt$par
    if(!(is.null(opt$hessian))){
      Asigma[param.Aidx] <- Sigma
    }
  }else if(svartype == "AB-model"){
    if(length(param.Aidx) > 0){
      Amat[param.Aidx] <- head(opt$par, nrow(param.Aidx))
      if(!(is.null(opt$hessian))){
        Asigma[param.Aidx] <- head(Sigma, nrow(param.Aidx))
      }
    } else {
      Amat <-  Amat
    }
    if(length(param.Bidx) > 0){
      Bmat[param.Bidx] <- tail(opt$par, nrow(param.Bidx))
      if(!(is.null(opt$hessian))){
        Bsigma[param.Bidx] <- tail(Sigma, nrow(param.Bidx))
      }
    } else {
      Bmat <-  Bmat
    }
  }
  colnames(Amat) <- colnames(x$y)
  rownames(Amat) <- colnames(Amat)
  colnames(Bmat) <- colnames(Amat)
  rownames(Bmat) <- colnames(Amat)
  colnames(Asigma) <- colnames(Amat)
  rownames(Asigma) <- colnames(Amat)
  colnames(Bsigma) <- colnames(Amat)
  rownames(Bsigma) <- colnames(Amat)  
  Sigma.U <- solve(Amat)%*%Bmat%*%t(Bmat)%*%t(solve(Amat))
  LRover <- NULL
  if(lrtest){
    degrees <- 2 * K^2 - params - 2 * K^2 + 0.5 * K * (K + 1)
    if(identical(degrees, 0)){
      warning(paste("The", svartype, "is just identified. No test possible."))
    } else {
      STATISTIC <- obs*(log(det(Sigma.U)) - log(det(sigma)))
      names(STATISTIC) <- "Chi^2"
      PARAMETER <- 2*K^2 - params - 2*K^2 + 0.5 * K * (K + 1)
      names(PARAMETER) <- "df"
      PVAL <- 1 - pchisq(STATISTIC, df = PARAMETER)
      METHOD <- "LR overidentification"
      LRover <- list(statistic = STATISTIC, parameter = PARAMETER, p.value = PVAL, method = METHOD, data.name = x$call$y)
      class(LRover) <- "htest"
    }
  }
  result <- list(A = Amat, Ase = Asigma, B = Bmat, Bse = Bsigma, LRIM = NULL, Sigma.U = Sigma.U*100, LR = LRover, opt = opt, start = start, type = svartype, var = x)
  class(result) <- "svarest"
  return(result)
}

".SVAR2" <-
function(x = x, Ra = NULL, Rb = NULL, ra = NULL, rb = NULL, start = NULL, max.iter = 100, conv.crit = 0.1e-6, maxls = 1.0, lrtest = TRUE){
  ##
  ## Some preliminary settings and computations
  ##
  obs <- x$obs
  Sigma <- crossprod(resid(x)) / obs
  n <- x$K
  nsq <- n^2
  ##
  ## Checking the SVAR-type and pre-setting vectors da and db
  ##
  if ((is.null(Ra)) && (is.null(Rb))){
    stop("\nAt least one matrix, either 'Ra' or 'Rb', must be non-null.\n")
  }
  if ((is.null(Ra)) && !(is.null(Rb))){
    bfree <- ncol(Rb)
    afree <- 0
    ra <- c(diag(n))
    svartype <- "B-model"
  } else if ((is.null(Rb)) && !(is.null(Ra))){
    afree <- ncol(Ra)
    bfree <- 0
    rb <- c(diag(n))
    svartype <- "A-model"
  } else {
    afree <- ncol(Ra)
    bfree <- ncol(Rb)
    svartype <- "AB-model"
  }
  l <- afree + bfree
  ##
  ## Defining the S matrix
  ##
  R <- matrix(0, nrow = 2*nsq, ncol = l)
  if(identical(afree, 0)){
    R[(nsq+1):(2*nsq), 1:l] <- Rb
  } else if(identical(bfree, 0)){
    R[1:nsq, 1:l] <- Ra
  } else if((!(is.null(afree)) && (!(is.null(bfree))))){
    R[1:nsq, 1:afree] <- Ra
    R[(nsq+1):(2*nsq), (afree+1):l] <- Rb
  }
  ##
  ## Defining the d vector
  ##
  r <- c(ra, rb)
  ##
  ## Commutation matrix Kkk and I_K^2 and I_K
  ##
  Kkk <- diag(nsq)[, c(sapply(1:n, function(i) seq(i, nsq, n)))]
  IK2 <- diag(nsq)
  IK <- diag(n)
  ##
  ## identification
  ##
  ifelse(is.null(start), gamma <- start <- rnorm(l), gamma <- start)
  vecab <- R %*% gamma + r
  A <- matrix(vecab[1:nsq], nrow = n, ncol = n)
  B <- matrix(vecab[(nsq + 1):(2*nsq)], nrow = n, ncol = n)
  v1 <- (IK2 + Kkk) %*% kronecker(t(solve(A) %*% B), solve(B))
  v2 <- -1.0 * (IK2 + Kkk) %*% kronecker(IK, solve(B))
  v <- cbind(v1, v2)
  idmat <- v %*% R
  ms <- t(v) %*% v
  auto <- eigen(ms)$values
  rni <- 0
  for(i in 1:l){
    if(auto[i] < 0.1e-10) rni <- rni + 1
  }
  if(identical(rni, 0)){
    if(identical(l, as.integer(n*(n + 1)/2))){
      ident <- paste("The", svartype, "is just identified.")
    } else {
      ident <- paste("The", svartype, "is over identified.")
    }
  } else {
    ident <- paste("The", svartype, "is unidentified. The non-identification rank is", rni, ".")
    stop(ident)
  }
  ##
  ## Scoring algorithm
  ##
  iters <- 0
  cvcrit <- conv.crit + 1.0
  while(cvcrit > conv.crit){
    z <- gamma
    vecab <- R %*% gamma + r
    A <- matrix(vecab[1:nsq], nrow = n, ncol = n)
    B <- matrix(vecab[(nsq + 1):(2*nsq)], nrow = n, ncol = n)
    Binv <- solve(B)
    Btinv <- solve(t(B))
    BinvA <- Binv %*% A
    infvecab.mat1 <- rbind(kronecker(solve(BinvA), Btinv), -1 * kronecker(IK, Btinv))
    infvecab.mat2 <- IK2 + Kkk
    infvecab.mat3 <- cbind(kronecker(t(solve(BinvA)), Binv), -1 * kronecker(IK, Binv))
    infvecab <- obs * (infvecab.mat1 %*% infvecab.mat2 %*% infvecab.mat3)
    infgamma <- t(R) %*% infvecab %*% R
    infgammainv <- solve(infgamma)
    scorevecBinvA <- obs * c(solve(t(BinvA))) - obs *(kronecker(Sigma, IK) %*% c(BinvA))
    scorevecAB.mat <- rbind(kronecker(IK, Btinv), -1.0 * kronecker(BinvA, Btinv))
    scorevecAB <- scorevecAB.mat %*% scorevecBinvA
    scoregamma <- t(R) %*% scorevecAB
    direction <- infgammainv %*% scoregamma
    length <- max(abs(direction))
    ifelse(length > maxls, lambda <- maxls/length, lambda <- 1.0)
    gamma <- gamma + lambda * direction    
    iters <- iters + 1
    z <- z - gamma
    cvcrit <- max(abs(z))
    if(iters >= max.iter){
      warning(paste("Convergence not achieved after", iters, "iterations. Convergence value:", cvcrit, "."))
      break
    }
  }
  vecab <- R %*% gamma + r
  colnames(A) <- colnames(x$y)
  rownames(A) <- colnames(A)
  colnames(B) <- colnames(A)
  rownames(B) <- colnames(A)
  ##
  ## Standard errors
  ##
  abSigma <- sqrt(diag((R %*% solve(infgamma) %*% t(R))))
  ASigma <- matrix(abSigma[1:nsq], nrow = n, ncol = n)
  BSigma <- matrix(abSigma[(nsq+1):(2*nsq)], nrow = n, ncol = n)
  colnames(ASigma) <- colnames(A)
  rownames(ASigma) <- rownames(A)
  colnames(BSigma) <- colnames(A)
  rownames(BSigma) <- rownames(A)
  Sigma.U <- solve(A) %*% B %*% t(B) %*% t(solve(A))
  ##
  ## LR Overidentification test
  ##
  LRover <- NULL
  if(lrtest){
    degrees <- 2 * n^2 - l - 2 * n^2 + 0.5 * n * (n + 1)
    if(identical(degrees, 0)){
      warning(paste("The", svartype, "is just identified. No test possible."))
    } else {
      rSigma <- solve(A) %*% B %*% t(B) %*% t(solve(A))
      det1 <- det(rSigma)
      det2 <- det(Sigma)
      STATISTIC <- (log(det1) - log(det2)) * obs
      names(STATISTIC) <- "Chi^2"
      PARAMETER <- degrees
      names(PARAMETER) <- "df"
      PVAL <- 1 - pchisq(STATISTIC, df = PARAMETER)
      METHOD <- "LR overidentification"
      LRover <- list(statistic = STATISTIC, parameter = PARAMETER, p.value = PVAL, method = METHOD, data.name = x$call$y)
      class(LRover) <- "htest"
    }
  }
  result <- list(A = A, Ase = ASigma, B = B, Bse = BSigma, LRIM = NULL, Sigma.U = Sigma.U * 100, LR = LRover, opt = NULL, start = start, type = svartype, var = x)
  class(result) <- "svarest"
  return(result)
}
