% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cast.R, R/type-bare.R
\name{vec_cast}
\alias{vec_cast}
\alias{vec_cast_common}
\alias{vec_restore}
\alias{vec_cast.logical}
\alias{vec_cast.integer}
\alias{vec_cast.double}
\alias{vec_cast.complex}
\alias{vec_cast.raw}
\alias{vec_cast.character}
\alias{vec_cast.list}
\title{Cast a vector to specified type}
\usage{
vec_cast(x, to, ..., x_arg = "x", to_arg = "to")

vec_cast_common(..., .to = NULL)

vec_restore(x, to, ..., n = NULL)

\method{vec_cast}{logical}(x, to, ...)

\method{vec_cast}{integer}(x, to, ...)

\method{vec_cast}{double}(x, to, ...)

\method{vec_cast}{complex}(x, to, ...)

\method{vec_cast}{raw}(x, to, ...)

\method{vec_cast}{character}(x, to, ...)

\method{vec_cast}{list}(x, to, ...)
}
\arguments{
\item{x}{Vectors to cast.}

\item{to, .to}{Type to cast to. If \code{NULL}, \code{x} will be returned as is.}

\item{...}{For \code{vec_cast_common()}, vectors to cast. For
\code{vec_cast()} and \code{vec_restore()}, these dots are only for future
extensions and should be empty.}

\item{x_arg, to_arg}{Argument names for \code{x} and \code{to}. These are used
in error messages to inform the user about the locations of
incompatible types (see \code{\link[=stop_incompatible_type]{stop_incompatible_type()}}).}

\item{n}{\Sexpr[results=rd, stage=render]{vctrs:::lifecycle("experimental")}
The total size to restore to. This is currently passed by
\code{vec_slice()} to solve edge cases arising in data frame
restoration. In most cases you don't need this information and
can safely ignore that argument. This parameter should be
considered internal and experimental, it might change in the
future.}
}
\value{
A vector the same length as \code{x} with the same type as \code{to},
or an error if the cast is not possible. An error is generated if
information is lost when casting between compatible types (i.e. when
there is no 1-to-1 mapping for a specific value).
}
\description{
\code{vec_cast()} provides general coercions from one type of vector to another,
and along with \code{\link[=vec_ptype2]{vec_ptype2()}} forms the foundation of the vctrs type system.
It should generally not be called by R users, but is important for R
developers. \code{vec_restore()} is designed specifically for casting a bare
vector to the original type; it's useful when relying \code{NextMethod()} for
the actual implementation. \code{vec_cast_common(...)} casts a collection to
vectors to the same type.
}
\section{Casting rules}{

Casting is more flexible than coercion, and allows for the possibility of
information loss. This diagram summarises possible coercions. \code{vec_cast()}
from any type connected to another type, provided that the arrows are
followed in only one direction. For example you can cast from logical to
character, and list to time, but you can not cast from logical to datetime.

\figure{cast.png}

Most casts are not symmetric: you can cast all integers to doubles, but you
can only cast a subset of doubles back to integers. If a cast is potentially
lossy, an error will be shown whenever an actual loss occurs.

The rules for coercing from a list are fairly strict: each component of the
list must be of length 1, and must be coercible to type \code{to}. This ensures
that a round-trip to and form list is possible, without opening the door
to very flexible list flattening (which should be the job of a more
specialised function).
}

\section{S3 dispatch}{

\code{vec_cast()} dispatches on both arguments because casting depends on both
the type of \code{x} and of \code{to}. This is implemented by having methods of
\code{vec_cast()}, e.g. \code{vec_cast.integer()} also be S3 generics, which call
e.g. \code{vec_cast.integer.double()}.

Note that \code{vec_cast()} dispatches on its second argument, so that the name
of the final method uses the same convention as \code{as.xyz()} methods, i.e.
\code{vec_cast.integer.double()} casts double to integers, in the same way
that \code{as.integer.double()} would.

Whenever you implement a \code{vec_cast.new_class()} generic/method,
make sure to always provide \code{vec_cast.new_class.default()} and
call \code{\link[=vec_default_cast]{vec_default_cast()}} from that method.

See \code{vignette("s3-vector")} for full details.
}

\section{Restoring attributes}{


A restore is a specialised type of cast, primarily used in
conjunction with \code{NextMethod()} or a C-level function that works on
the underlying data structure. A \code{vec_restore()} method can make
the following assumptions about \code{x}:
\itemize{
\item It has the correct type.
\item It has the correct names.
\item It has the correct \code{dim} and \code{dimnames} attributes.
\item It is unclassed. This way you can call vctrs generics with \code{x}
without triggering an infinite loop of restoration.
}

The length may be different (for example after \code{\link[=vec_slice]{vec_slice()}} has
been called), and all other attributes may have been lost. The
method should restore all attributes so that after restoration,
\code{vec_restore(vec_data(x), x)} yields \code{x}.

To understand the difference between \code{vec_cast()} and \code{vec_restore()}
think about factors: it doesn't make sense to cast an integer to a factor,
but if \code{NextMethod()} or another low-level function has stripped attributes,
you still need to be able to restore them.

The default method copies across all attributes so you only need to
provide your own method if your attributes require special care
(i.e. they are dependent on the data in some way). When implementing
your own method, bear in mind that many R users add attributes to track
additional metadata that is important to them, so you should preserve any
attributes that don't require special handling for your class.
}

\examples{
# x is a double, but no information is lost
vec_cast(1, integer())

# When information is lost the cast fails
try(vec_cast(c(1, 1.5), integer()))
try(vec_cast(c(1, 2), logical()))

# You can suppress this error and get the partial results
allow_lossy_cast(vec_cast(c(1, 1.5), integer()))
allow_lossy_cast(vec_cast(c(1, 2), logical()))

# By default this suppress all lossy cast errors without
# distinction, but you can be specific about what cast is allowed
# by supplying prototypes
allow_lossy_cast(vec_cast(c(1, 1.5), integer()), to_ptype = integer())
try(allow_lossy_cast(vec_cast(c(1, 2), logical()), to_ptype = integer()))

# No sensible coercion is possible so an error is generated
try(vec_cast(1.5, factor("a")))

# Cast to common type
vec_cast_common(factor("a"), factor(c("a", "b")))
vec_cast_common(factor("a"), Sys.Date(), .to = list())
}
\keyword{internal}
