% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/galaxy.trimfill.R
\name{galaxy.trimfill}
\alias{galaxy.trimfill}
\title{Bivariate trim&fill method}
\usage{
galaxy.trimfill(
  y1,
  v1,
  y2,
  v2,
  n.grid = 12,
  angle,
  estimator,
  side,
  rho = 0,
  method = "mm",
  method.uni = "DL",
  maxiter = 20,
  var.names = c("y1", "y2"),
  scale = 0.02,
  verbose = FALSE
)
}
\arguments{
\item{y1}{vector of the effect size estimates of the first outcome}

\item{v1}{estimated variance of \code{y1}}

\item{y2}{vector of the effect size estimates of the second outcome}

\item{v2}{estimated variance of \code{y2}}

\item{n.grid}{number of grid (equally spaced) candidate directions that the optimal projection direction are searched among, see Details}

\item{angle}{angles of candidate projection directions not by grid, this will overwrite n.grid}

\item{estimator}{estimator used for the number of trimmed studies in univariate T&F on the projected studies, one of c('R0', 'L0', 'Q0')}

\item{side}{either "left" or "right", indicating on which side of the galaxy plot 
the missing studies should be imputed. If null determined by the univariate T&F}

\item{rho}{correlation between y1 and y2 when computing the variance of the projected studies. Default is the estimated cor(y1, y2)}

\item{method}{method to estimate the center for the bivariate outcomes. Default is 'mm', i.e. random-effects model}

\item{method.uni}{method to estimate the center for the univariate projected studies using a univariate T&F procedure. Default is 'DL', i.e. fixed-effects model}

\item{maxiter}{max number of iterations used in the univariate T&F. Default is 20.}

\item{var.names}{names of the two outcomes used in the galaxy plot (if plotted). Default is c('y1', 'y2')}

\item{scale}{constant scale for plotting the galaxy plot for the bivariate studies, Default is 0.02.}

\item{verbose}{plot the galaxy plot? Default is FALSE.}
}
\value{
List with component:

res  a data.frame of 9 columns and n.grid rows. Each row is the result for projection along one candidate grid direction, and the columns are named: 
               'y1.c', 'y2.c' for projected bivariate center, 'y1.f', 'y2.f' for bivariate center using filled studies,
              'k0', 'se.k0' for estimated number of trimmed studies and its standard error, 'se.y1.f', 'se.y2.f' for standard errors of 'y1.f', 'y2.f', 
              'side.left' for the estimated side

ID.trim list of vectors of ids of studies been trimmed along each of the candidate direction.
}
\description{
Bivariate T&F method accounting for small-study effects in bivariate meta-analysis, based on symmetry of the galaxy plot.
}
\details{
The bivariate T&F method assumes studies are suppressed based on a weighted sum of the two outcomes, i.e. the studies with smallest values 
          of z_i = c_1 * y_1i + c_2 * y_2i, i=1,...,N  are suppressed. We use a searching algorithm to find the optimal ratio of c_1  and c_2 (i.e. a direction), 
          which gives the most trimmed studies. This is based on the observation that the closer a direction is to the truth, the more studies 
          are expected to be trimmed along that direction. We set a sequence of equally-spaced candidate directions with angle a_m = m*pi/M, 
          and (c_1, c_2) = (cos(a_m), sin(a_m)), m=1,...,M.
}
\examples{
\dontrun{ 
require(MASS)
require(mvmeta)
require(metafor)
set.seed(123)
mydata <- dat.gen(m.o=50, m.m=20,       # # observed studies, # missing studies
                  s.m= c(0.5, 0.5),     #  c(mean(s1), mean(s2))
                  angle.LC = pi/4,      # suppress line direction 
                  mybeta=c(2,2),        # true effect size
                  tau.sq=c(0.1, 0.1),   # true between-study var
                  rho.w=0.5, rho.b=0.5, # true within-study and between-study corr
                  s.min = 0.1,          # s1i ~ Unif(s.min, 2*s.m[1]-s.min) 
                  verbose = TRUE)

y1 <- mydata$mydat.sps$y1
y2 <- mydata$mydat.sps$y2
v1 <- mydata$mydat.sps$s1^2
v2 <- mydata$mydat.sps$s2^2

## unadjusted est 
mv_obs <- mvmeta(cbind(y1, y2), cbind(v1, v2), method='mm')
c(mv_obs$coef)
# 2.142687 2.237741

estimator <- 'R0' 
## univariate T&F based on y1 or y2
y1.rma <- rma(y1, v1, method='FE')
y2.rma <- rma(y2, v2, method='FE')
y1.tf <- trimfill.rma(y1.rma, estimator = estimator, method.fill = 'DL') 
y2.tf <- trimfill.rma(y2.rma, estimator = estimator, method.fill = 'DL') 
c(y1.tf$beta, y2.tf$beta)
# 2.122231 2.181333
c(y1.tf$k0, y2.tf$k0)
# 2 8

## bivariate T&F method (based on galaxy plot)
tf.grid <- galaxy.trimfill(y1, v1, y2, v2, n.grid = 12,  
                           estimator=estimator, side='left',
                           method.uni = 'FE',
                           method = 'mm', 
                           rho=0.5, maxiter=100, verbose=FALSE) 
tf.grid$res
tf.grid$res[which(tf.grid$res$k0==max(tf.grid$res$k0)),3:5] 
#     y1.f     y2.f k0
# 2.053306 2.162347 14  

## less bias by the proposed bivariate T&F method
rbind(true = c(2,2),
      unadjusted=c(mv_obs$coef), 
      tf.uni = c(y1.tf$beta, y2.tf$beta),
      tf.biv = tf.grid$res[which(tf.grid$res$k0==max(tf.grid$res$k0)),3:4])

## unlike the univariate T&Fs, biv T&F obtains one estimate of # missing studies
c(k0.true = 20,
  k0.tf.uni.y1 = y1.tf$k0, 
  k0.tf.uni.y2 = y2.tf$k0, 
  k0.tf.biv = tf.grid$res[which(tf.grid$res$k0==max(tf.grid$res$k0)),5])
# k0.true k0.tf.uni.y1 k0.tf.uni.y2    k0.tf.biv 
# 20            2            8           14
}
}
\references{
Luo C, Marks-Anglin AK, Duan R, Lin L, Hong C, Chu H, Chen Y. Accounting for small-study effects 
               using a bivariate trim and fill meta-analysis procedure. medRxiv. 2020 Jan 1.
}
\author{
Chongliang Luo, Yong Chen
}
