% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prob-roc_auc.R
\name{roc_auc}
\alias{roc_auc}
\alias{roc_auc.data.frame}
\alias{roc_auc_vec}
\title{Area under the receiver operator curve}
\usage{
roc_auc(data, ...)

\method{roc_auc}{data.frame}(
  data,
  truth,
  ...,
  estimator = NULL,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL,
  options = list()
)

roc_auc_vec(
  truth,
  estimate,
  estimator = NULL,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL,
  options = list(),
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} containing the columns specified by \code{truth} and
\code{...}.}

\item{...}{A set of unquoted column names or one or more
\code{dplyr} selector functions to choose which variables contain the
class probabilities. If \code{truth} is binary, only 1 column should be selected,
and it should correspond to the value of \code{event_level}. Otherwise, there
should be as many columns as factor levels of \code{truth} and the ordering of
the columns should be the same as the factor levels of \code{truth}.}

\item{truth}{The column identifier for the true class results
(that is a \code{factor}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:topic-inject]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{factor} vector.}

\item{estimator}{One of \code{"binary"}, \code{"hand_till"}, \code{"macro"}, or
\code{"macro_weighted"} to specify the type of averaging to be done. \code{"binary"}
is only relevant for the two class case. The others are general methods for
calculating multiclass metrics. The default will automatically choose
\code{"binary"} if \code{truth} is binary, \code{"hand_till"} if \code{truth} has >2 levels and
\code{case_weights} isn't specified, or \code{"macro"} if \code{truth} has >2 levels and
\code{case_weights} is specified (in which case \code{"hand_till"} isn't
well-defined).}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}

\item{event_level}{A single string. Either \code{"first"} or \code{"second"} to specify
which level of \code{truth} to consider as the "event". This argument is only
applicable when \code{estimator = "binary"}. The default uses an internal helper
that defaults to \code{"first"}.}

\item{case_weights}{The optional column identifier for case weights.
This should be an unquoted column name that evaluates to a numeric column
in \code{data}. For \verb{_vec()} functions, a numeric vector,
\code{\link[hardhat:importance_weights]{hardhat::importance_weights()}}, or \code{\link[hardhat:frequency_weights]{hardhat::frequency_weights()}}.}

\item{options}{\verb{[deprecated]}

No longer supported as of yardstick 1.0.0. If you pass something here it
will be ignored with a warning.

Previously, these were options passed on to \code{pROC::roc()}. If you need
support for this, use the pROC package directly.}

\item{estimate}{If \code{truth} is binary, a numeric vector of class probabilities
corresponding to the "relevant" class. Otherwise, a matrix with as many
columns as factor levels of \code{truth}. \emph{It is assumed that these are in the
same order as the levels of \code{truth}.}}
}
\value{
A \code{tibble} with columns \code{.metric}, \code{.estimator},
and \code{.estimate} and 1 row of values.

For grouped data frames, the number of rows returned will be the same as
the number of groups.

For \code{roc_auc_vec()}, a single \code{numeric} value (or \code{NA}).
}
\description{
\code{roc_auc()} is a metric that computes the area under the ROC curve. See
\code{\link[=roc_curve]{roc_curve()}} for the full curve.
}
\details{
Generally, an ROC AUC value is between \code{0.5} and \code{1}, with \code{1} being a
perfect prediction model. If your value is between \code{0} and \code{0.5}, then
this implies that you have meaningful information in your model, but it
is being applied incorrectly because doing the opposite of what the model
predicts would result in an AUC \verb{>0.5}.

Note that you can't combine \code{estimator = "hand_till"} with \code{case_weights}.
}
\section{Relevant Level}{


There is no common convention on which factor level should
automatically be considered the "event" or "positive" result
when computing binary classification metrics. In \code{yardstick}, the default
is to use the \emph{first} level. To alter this, change the argument
\code{event_level} to \code{"second"} to consider the \emph{last} level of the factor the
level of interest. For multiclass extensions involving one-vs-all
comparisons (such as macro averaging), this option is ignored and
the "one" level is always the relevant result.
}

\section{Multiclass}{

The default multiclass method for computing \code{roc_auc()} is to use the
method from Hand, Till, (2001). Unlike macro-averaging, this method is
insensitive to class distributions like the binary ROC AUC case.
Additionally, while other multiclass techniques will return \code{NA} if any
levels in \code{truth} occur zero times in the actual data, the Hand-Till method
will simply ignore those levels in the averaging calculation, with a warning.

Macro and macro-weighted averaging are still provided, even though they are
not the default. In fact, macro-weighted averaging corresponds to the same
definition of multiclass AUC given by Provost and Domingos (2001).
}

\examples{
# ---------------------------------------------------------------------------
# Two class example

# `truth` is a 2 level factor. The first level is `"Class1"`, which is the
# "event of interest" by default in yardstick. See the Relevant Level
# section above.
data(two_class_example)

# Binary metrics using class probabilities take a factor `truth` column,
# and a single class probability column containing the probabilities of
# the event of interest. Here, since `"Class1"` is the first level of
# `"truth"`, it is the event of interest and we pass in probabilities for it.
roc_auc(two_class_example, truth, Class1)

# ---------------------------------------------------------------------------
# Multiclass example

# `obs` is a 4 level factor. The first level is `"VF"`, which is the
# "event of interest" by default in yardstick. See the Relevant Level
# section above.
data(hpc_cv)

# You can use the col1:colN tidyselect syntax
library(dplyr)
hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  roc_auc(obs, VF:L)

# Change the first level of `obs` from `"VF"` to `"M"` to alter the
# event of interest. The class probability columns should be supplied
# in the same order as the levels.
hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  mutate(obs = relevel(obs, "M")) \%>\%
  roc_auc(obs, M, VF:L)

# Groups are respected
hpc_cv \%>\%
  group_by(Resample) \%>\%
  roc_auc(obs, VF:L)

# Weighted macro averaging
hpc_cv \%>\%
  group_by(Resample) \%>\%
  roc_auc(obs, VF:L, estimator = "macro_weighted")

# Vector version
# Supply a matrix of class probabilities
fold1 <- hpc_cv \%>\%
  filter(Resample == "Fold01")

roc_auc_vec(
   truth = fold1$obs,
   matrix(
     c(fold1$VF, fold1$F, fold1$M, fold1$L),
     ncol = 4
   )
)

}
\references{
Hand, Till (2001). "A Simple Generalisation of the Area Under the
ROC Curve for Multiple Class Classification Problems". \emph{Machine Learning}.
Vol 45, Iss 2, pp 171-186.

Fawcett (2005). "An introduction to ROC analysis". \emph{Pattern Recognition
Letters}. 27 (2006), pp 861-874.

Provost, F., Domingos, P., 2001. "Well-trained PETs: Improving probability
estimation trees", CeDER Working Paper #IS-00-04, Stern School of Business,
New York University, NY, NY 10012.
}
\seealso{
\code{\link[=roc_curve]{roc_curve()}} for computing the full ROC curve.

Other class probability metrics: 
\code{\link{average_precision}()},
\code{\link{brier_class}()},
\code{\link{classification_cost}()},
\code{\link{gain_capture}()},
\code{\link{mn_log_loss}()},
\code{\link{pr_auc}()},
\code{\link{roc_aunp}()},
\code{\link{roc_aunu}()}
}
\author{
Max Kuhn
}
\concept{class probability metrics}
