% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/admm.lasso.R
\name{admm.lasso}
\alias{admm.lasso}
\title{LASSO}
\usage{
admm.lasso(A, b, lambda = 1, xinit = NA, rho = 1, alpha = 1,
  abstol = 1e-04, reltol = 0.01, maxiter = 1000)
}
\arguments{
\item{A}{an \eqn{(m\times n)} regressor matrix}

\item{b}{a length-\eqn{m} response vector}

\item{lambda}{a regularization parameber}

\item{xinit}{a length-\eqn{n} vector for initial value}

\item{rho}{an augmented Lagrangian parameter}

\item{alpha}{an overrelaxation parameter in [1,2]}

\item{abstol}{absolute tolerance stopping criterion}

\item{reltol}{relative tolerance stopping criterion}

\item{maxiter}{maximum number of iterations}
}
\value{
a named list containing \describe{
\item{x}{a length-\eqn{n} solution vector}
\item{history}{dataframe recording iteration numerics. See the section for more details.}
}
}
\description{
LASSO, or L1-regularized regression, is an optimization problem to solve
\deqn{min_x ~ \frac{1}{2}\|Ax-b\|_2^2 + \lambda \|x\|_1}
for sparsifying the coefficient vector \eqn{x}.
The implementation is borrowed from Stephen Boyd's
\href{http://stanford.edu/~boyd/papers/admm/lasso/lasso.html}{MATLAB code}.
}
\section{Iteration History}{

When you run the algorithm, output returns not only the solution, but also the iteration history recording
following fields over iterates,
\describe{
\item{objval}{object (cost) function value}
\item{r_norm}{norm of primal residual}
\item{s_norm}{norm of dual residual}
\item{eps_pri}{feasibility tolerance for primal feasibility condition}
\item{eps_dual}{feasibility tolerance for dual feasibility condition}
}
In accordance with the paper, iteration stops when both \code{r_norm} and \code{s_norm} values
become smaller than \code{eps_pri} and \code{eps_dual}, respectively.
}

\examples{
## generate sample data
m = 500
n = 1000
p = 0.1   # percentange of non-zero elements

x0 = matrix(Matrix::rsparsematrix(n,1,p))
A  = matrix(rnorm(m*n),nrow=m)
for (i in 1:ncol(A)){
  A[,i] = A[,i]/sqrt(sum(A[,i]*A[,i]))
}
b = A\%*\%x0 + sqrt(0.001)*matrix(rnorm(m))

## set regularization lambda value
lambda = 0.1*Matrix::norm(t(A)\%*\%b, 'I')

## run example
output = admm.lasso(A, b, lambda)

## report convergence plot
niter  = length(output$history$s_norm)
par(mfrow=c(1,3))
plot(1:niter, output$history$objval, "b", main="cost function")
plot(1:niter, output$history$r_norm, "b", main="primal residual")
plot(1:niter, output$history$s_norm, "b", main="dual residual")

}
\references{
\insertRef{tibshirani_regression_1996}{ADMM}
}
