\name{summedCalibrator} 
\alias{summedCalibrator} 
\title{Generates a summed probability distribution (SPD) of calibrated dates} 
\description{Generates a Summed Probability Distribution (SPD). Handles both C14 and other date types e.g., thermoluminescence.}
\usage{summedCalibrator(data, CalArray, normalise = 'standard', checks = TRUE)}
\arguments{
  \item{data}{
	A dataframe of 14C dates. Requires 'age' and 'sd', and at least one of 'site' and 'phase'. Optional 'datingType' to include '14C' and anything else.
	}
  \item{CalArray}{
	A 2D probability array of the calibration curve generated by \code{\link[=makeCalArray]{makeCalArray}} containing row names and column names. 
	}
  \item{normalise}{
	One of 'none', 'standard', or 'full'. 
	}
  \item{checks}{
	Logical, performs various data checks if TRUE. Can be useful to set FALSE to avoid repetitive warnings if run in a loop. 
	}
}
\value{
	Returns a single-column data frame of SPD probabilities. Row names are the calendar years.
	}
\details{
Uses CalArray once to simultaneously calibrate and sum all 14C dates in data. The result is equivalent to calibrating each date, then summing. 

Optionally 'datingType' can be provided in the data. Only '14C' will be calibrated in the usual way, anything else is assumed to be provided in calendar time. If 'datingType' is not provided, all dates are assumed to be 14C.

If normalise is 'none', the output PD has an area equal to the total number of samples within the date range. This option is rarely required, but can be useful for example when plotting several SPDs and would also like to illustrate the relative magnitude of different datasets. However, if the date range used to generate CalArray does not encompass the entire dataset, some dates will have some probability mass outside the date range. Therefore the total probability can be non-integer and less than the sample size. 

If normalise is 'standard', the output PD is normalised by the number of samples, giving an area equal to 1 provided all samples are within the date range. However, if the date range does not encompass all samples, some will have some probability mass outside the date range, resulting in the SPD area being less than 1.

If normalise is 'full', the output PD has an area equal to 1. An appropriate use includes SPD simulation testing, where it is important to ensure each simulation has the same area. In contrast, it would be absurd to apply this full normalisation to the tiny tail of a single date that is otherwise mostly outside the date range.
	}
\examples{
	# SPD of three 14C dates
	CalArray <- makeCalArray(intcal20, calrange=c(9000,10650), inc=1 )
	data <- data.frame(
		age = c(8350,8900,9350), 
		sd = rep(50,3),
		datingType = rep('14C',3)
		)


	# with the default normalisation the SPD area is a little under 1
	# since one date is slighly outside the date range
	SPD <- summedCalibrator(data, CalArray)
	plotPD(SPD)
	sum(SPD) 

	# without normalisation the total area is a little under 3
	SPD <- summedCalibrator(data, CalArray, normalise='none')
	plotPD(SPD)
	sum(SPD)

	# with full normalisation the total area is exactly 1
	SPD <- summedCalibrator(data, CalArray, normalise='full')
	plotPD(SPD)
	sum(SPD)
	}
